<?php

declare(strict_types=1);

namespace Avodel\FrameworkBundle\DependencyInjection;

use Avodel\Clock\Clock;
use Avodel\Logger\ContextAwareExceptionInterface;
use Avodel\JwtHttpClient\JwtFetcher\UserCredentialsJwtFetcher;
use Avodel\JwtHttpClient\JwtHttpClient;
use Behat\Behat\Context\Context;
use Avodel\Uuid\UuidGeneratorInterface;
use Avodel\Healthcheck\Controller\HealthcheckController;
use Avodel\MigrationLock\LockDoctrineMigrationListener;
use Symfony\Component\Config\FileLocator;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Definition;
use Symfony\Component\DependencyInjection\Extension\Extension;
use Symfony\Component\DependencyInjection\Loader\YamlFileLoader;
use Symfony\Component\DependencyInjection\Reference;
use Symfony\Component\Lock\LockInterface;

class AvodelFrameworkExtension extends Extension
{
    public function load(array $configs, ContainerBuilder $container): void
    {
        $configuration = new Configuration();
        $config = $this->processConfiguration($configuration, $configs);

        $this->configureLockDbMigrations($config, $container);
        $this->configureHealthCheck($config, $container);
        $this->configureClock($config, $container);
        $this->configureUuid($config, $container);
        $this->configureLogger($config, $container);
        $this->configureJwtHttpClient($config, $container);
        $this->configureHostName($config, $container);
    }

    private function configureLockDbMigrations(array $config, ContainerBuilder $container): void
    {
        $migrationsLockEnabled = $config['lock_db_migration'] ?? false;
        if ($migrationsLockEnabled) {
            if (!class_exists(LockDoctrineMigrationListener::class)) {
                throw new \LogicException('To enable the migrations locking feature you must install the avodel/migration-lock package. Try running "composer require avodel/migration-lock".');
            }
            if (!interface_exists(LockInterface::class)) {
                throw new \LogicException('To enable the migrations locking feature you must install the symfony/lock package. Try running "composer require symfony/lock".');
            }

            $subscriberDefinition = new Definition(LockDoctrineMigrationListener::class);
            $subscriberDefinition->setAutoconfigured(true);
            $subscriberDefinition->setAutowired(true);
            $subscriberDefinition->setArgument('$prefix', $config['application_name']);
            $container->setDefinition(LockDoctrineMigrationListener::class, $subscriberDefinition);
        }
    }

    private function configureHealthCheck(array $config, ContainerBuilder $container): void
    {
        $healthcheckEnabled = $config['healthcheck'] ?? false;
        if ($healthcheckEnabled) {
            if (!class_exists(HealthcheckController::class)) {
                throw new \LogicException('To enable healthcheck feature you must install the avodel/healthcheck package. Try running "composer require avodel/healthcheck".');
            }

            $controllerDefinition = new Definition(HealthcheckController::class);
            $controllerDefinition->addTag('controller.service_arguments');
            $container->setDefinition(HealthcheckController::class, $controllerDefinition);
        }
    }

    private function configureClock(array $config, ContainerBuilder $container): void
    {
        if (array_key_exists('clock', $config)) {
            if (!class_exists(Clock::class)) {
                throw new \LogicException('To enable the clock feature you must install the avodel/clock package. Try running "composer require avodel/clock".');
            }

            $loader = new YamlFileLoader($container, new FileLocator(__DIR__ . '/../Resources/Clock/config/'));
            $loader->load('services.yaml');

            $behatEnabled = $config['clock']['behat'] ?? false;
            if ($behatEnabled) {
                if (!interface_exists(Context::class)) {
                    throw new \LogicException('To enable behat in clock feature you must install the behat/behat package. Try running "composer require behat/behat".');
                }

                $loader->load('behat_services.yaml');
            }
        }
    }

    private function configureUuid(array $config, ContainerBuilder $container): void
    {
        if (array_key_exists('uuid', $config)) {
            if (!interface_exists(UuidGeneratorInterface::class)) {
                throw new \LogicException('To enable the uuid feature you must install the avodel/uuid package. Try running "composer require avodel/uuid".');
            }

            $loader = new YamlFileLoader($container, new FileLocator(__DIR__ . '/../Resources/Uuid/config/'));
            $loader->load('services.yaml');

            $behatEnabled = $config['uuid']['behat'] ?? false;
            if ($behatEnabled) {
                if (!interface_exists(Context::class)) {
                    throw new \LogicException('To enable behat in uuid feature you must install the behat/behat package. Try running "composer require behat/behat".');
                }

                $loader->load('behat_services.yaml');
            }
        }
    }

    private function configureLogger(array $config, ContainerBuilder $container): void
    {
        if (array_key_exists('logger', $config)) {
            if (!interface_exists(ContextAwareExceptionInterface::class)) {
                throw new \LogicException('To enable the logger feature you must install the avodel/logger package. Try running "composer require avodel/logger".');
            }

            $loader = new YamlFileLoader($container, new FileLocator(__DIR__ . '/../Resources/Logger/config/'));
            $loader->load('services.yaml');

            $config = $config['logger'];
            $container->setParameter('avodel.logger.elasticsearch.url', $config['elasticsearch']['url']);
            $container->setParameter('avodel.logger.elasticsearch.username', $config['elasticsearch']['username']);
            $container->setParameter('avodel.logger.elasticsearch.password', $config['elasticsearch']['password']);
            $container->setParameter('avodel.logger.elasticsearch.index', $config['elasticsearch']['index']);
            $container->setParameter('avodel.logger.elasticsearch.enabled', $config['elasticsearch']['enabled']);
            $container->setParameter('avodel.logger.project_dir', $config['project_dir']);
            $container->setParameter('avodel.logger.exclusions', $config['exclusions']);
        }
    }

    private function configureJwtHttpClient(array $config, ContainerBuilder $container): void
    {
        if (array_key_exists('jwt_http_client', $config)) {
            if(!class_exists(JwtHttpClient::class)) {
                throw new \LogicException('To enable the jwt_http_client feature you must install the avodel/jwt-http-client package. Try running "composer require avodel/jwt-http-client".');
            }

            $config = $config['jwt_http_client'];
            foreach ($config['clients'] as $clientName => $clientConfig) {
                $fetcherId = 'avodel.jwt_http_client.jwt_fetcher.' . $clientName;
                $clientId = 'avodel.jwt_http_client.' . $clientName;

                $jwtFetcher = new Definition();
                $jwtFetcher->setClass(UserCredentialsJwtFetcher::class);
                $jwtFetcher->setArguments([
                    '$httpClient' => new Reference($clientId . '.inner'),
                    '$username' => $clientConfig['auth']['username'],
                    '$password' => $clientConfig['auth']['password'],
                    '$cache' => new Reference($clientConfig['jwt_storage']),
                    '$logger' => new Reference('logger'),
                ]);
                $container->setDefinition($fetcherId, $jwtFetcher);

                $definition = new Definition(JwtHttpClient::class);
                $definition->setDecoratedService($clientConfig['http_client']);
                $definition->setArguments([
                    '$decorated' => new Reference($clientId . '.inner'),
                    '$jwtFetcher' => new Reference($fetcherId),
                ]);
                $container->setDefinition($clientId, $definition);
            }
        }
    }

    private function configureHostName(array $config, ContainerBuilder $container): void
    {
        $env = $container->getParameter('kernel.environment');

        if ($env === 'prod') {
            $container->setParameter('avodel.hostname', gethostname());
            return;
        }

        $container->setParameter('avodel.hostname', $config['application_name'] . "_$env");
    }
}
