# Avodel Framework Bundle

## Overview

Avodel Framework Bundle is a Symfony bundle that serves as a unified configuration hub for the Avodel component ecosystem. It aggregates and configures multiple specialized packages (clock, UUID, logging, JWT authentication, testing utilities) through a single, centralized configuration entry point. The bundle follows a plugin architecture where each feature is conditionally loaded based on configuration presence, preventing unnecessary dependencies and tight coupling between components.

## Business Logic

### Core Concept

**Problem:** Symfony applications using the Avodel ecosystem require multiple packages for cross-cutting concerns (logging, authentication, testing). Each package needs separate configuration, service registration, and dependency management.

**Solution:** This bundle provides a single `avodel_framework` configuration node that orchestrates all Avodel components. Features are loaded on-demand when their configuration section is present, with fail-fast validation that provides helpful error messages when required packages are missing.

### Main Process

```
1. BUNDLE LOAD (AvodelFrameworkExtension)
   │
   ├── Parse avodel_framework configuration
   ├── Validate application_name (required)
   └── Process each feature section conditionally
         │
2. FEATURE ACTIVATION (per-section)
   │
   ├── clock section present?
   │   ├── YES → Load Clock services + optional Behat stubs
   │   └── NO  → Skip
   │
   ├── uuid section present?
   │   ├── YES → Load UUID generator + optional Behat stubs
   │   └── NO  → Skip
   │
   ├── logger section present?
   │   ├── YES → Load logging services + Elasticsearch handler
   │   └── NO  → Skip
   │
   ├── jwt_http_client section present?
   │   ├── YES → Decorate HTTP clients with JWT auth
   │   └── NO  → Skip
   │
   ├── lock_db_migration: true?
   │   ├── YES → Register migration lock listener
   │   └── NO  → Skip
   │
   ├── healthcheck: true?
   │   ├── YES → Register health check controller
   │   └── NO  → Skip
   │
   └── symfony_behat_api section present?
       ├── YES → Load enabled Behat contexts
       └── NO  → Skip

3. HOSTNAME RESOLUTION (always)
   │
   └── Register avodel.hostname service
       ├── Production env → OS hostname (gethostname())
       └── Other envs    → "{application_name}_{environment}"
```

### Key Algorithms

**JWT HTTP Client Decoration:**
```
For each configured client:
    │
    ├── Create JwtFetcher with credentials
    ├── Create JwtHttpClient decorator
    └── Decorate original HTTP client service
        │
        └── All requests automatically include JWT bearer token
            Token is cached and refreshed transparently
```

**Hostname Resolution Logic:**
| Environment | Result |
|-------------|--------|
| `prod` | System hostname via `gethostname()` |
| `dev`, `test`, `staging`, etc. | `{application_name}_{environment}` |

### Feature Dependencies

| Feature | Required Package | Optional Package |
|---------|------------------|------------------|
| `clock` | `avodel/clock` | `behat/behat` (for stubs) |
| `uuid` | `avodel/uuid` | `behat/behat` (for stubs) |
| `logger` | `avodel/logger` | — |
| `jwt_http_client` | `avodel/jwt-http-client` | — |
| `lock_db_migration` | `avodel/migration-lock`, `symfony/lock` | — |
| `healthcheck` | `avodel/healthcheck` | — |
| `symfony_behat_api` | `avodel/symfony-behat-api` | — |

## Data Model

This bundle does not define entities. It coordinates external packages:

```
AvodelFrameworkBundle (configuration hub)
├── Clock Component
│   └── ClockInterface → Clock implementation
├── UUID Component
│   └── UuidGeneratorInterface → UuidGenerator
├── Logger Component
│   ├── KibanaFormatter
│   ├── ExceptionProcessor
│   ├── FileLineProcessor
│   ├── ExclusionsHandler
│   └── ElasticsearchLogstashHandler
├── JWT HTTP Client
│   ├── JwtHttpClient (decorator)
│   └── UserCredentialsJwtFetcher
├── Migration Lock
│   └── LockDoctrineMigrationListener
├── Healthcheck
│   └── HealthcheckController
└── Behat API Contexts
    ├── RestApiContext
    ├── CliContext
    ├── DoctrineORMContext
    ├── HttpClientContext
    └── JwtContext
```

See external package repositories for implementation details.

## External Dependencies

| Package | Purpose | Type |
|---------|---------|------|
| `avodel/clock` | PSR-20 Clock interface | Library |
| `avodel/uuid` | UUID generation | Library |
| `avodel/logger` | Enhanced Monolog logging | Library |
| `avodel/jwt-http-client` | JWT-authenticated HTTP clients | Library |
| `avodel/healthcheck` | Health check endpoint | Library |
| `avodel/migration-lock` | Concurrent migration prevention | Library |
| `avodel/symfony-behat-api` | Behat testing contexts | Library (dev) |
| Elasticsearch | Log aggregation (optional) | External Service |

## Bundle Configuration

```yaml
avodel_framework:
    application_name: string          # Required - used for hostname and migration locks

    lock_db_migration: bool           # default: false
    healthcheck: bool                 # default: false

    clock:                            # Optional section - enables clock component
        behat: bool                   # default: false

    uuid:                             # Optional section - enables UUID component
        behat: bool                   # default: false

    logger:                           # Optional section - enables logger component
        project_dir: string           # Required - for stack trace paths
        elasticsearch:
            enabled: bool             # default: false
            url: string               # Required
            username: string          # Required
            password: string          # Required
            index: string             # Required
        exclusions:                   # Array of log filters
            - channel: string         # Monolog channel name
              message: string         # Regex pattern to exclude

    jwt_http_client:                  # Optional section
        clients:
            client_name:              # Service name pattern: avodel.jwt_http_client.{name}
                http_client: string   # Required - service ID to decorate
                jwt_storage: string   # Required - cache service ID
                auth:
                    username: string  # Required
                    password: string  # Required

    symfony_behat_api:                # Optional section
        cli: bool                     # default: false
        doctrine: bool                # default: false
        http_client: bool             # default: false
        jwt: bool                     # default: false
        rest_api: bool                # default: false

    swagger:                          # Optional section
        output_dir: string            # default: '.generated'
        enable_clients: bool          # default: false
```

## Registered Services

| Service ID | Class/Type | When Registered |
|------------|------------|-----------------|
| `avodel.hostname` | `string` | Always |
| `Psr\Clock\ClockInterface` | `Avodel\Clock\Clock` | `clock` section present |
| `Avodel\Uuid\UuidGeneratorInterface` | `Avodel\Uuid\UuidGenerator` | `uuid` section present |
| `avodel.jwt_http_client.{name}` | `JwtHttpClient` | Per client in `jwt_http_client.clients` |
| `avodel.logger.elasticsearch_logstash.handler` | `Monolog\Handler\Handler` | `logger` section present |
| `Avodel\Healthcheck\Controller\HealthcheckController` | Controller | `healthcheck: true` |
| `Avodel\MigrationLock\LockDoctrineMigrationListener` | Event Subscriber | `lock_db_migration: true` |

## Behat Testing Contexts

The `symfony_behat_api` section enables Behat contexts for testing various aspects of Symfony applications. Each context requires enabling the corresponding flag in configuration and registering the context in `behat.yml.dist`.

### REST API Context

Provides step definitions for testing REST API endpoints.

**Configuration** (`config/packages/test/avodel_framework.yaml`):
```yaml
when@test:
  avodel_framework:
    symfony_behat_api:
      rest_api: true
```

**Behat registration** (`behat.yml.dist`):
```yaml
default:
    suites:
        default:
            contexts:
                - Avodel\SymfonyBehatApi\RestApi\RestApiContext
```

### Doctrine ORM Context

Provides step definitions for testing database operations and entity state.

**Configuration** (`config/packages/test/avodel_framework.yaml`):
```yaml
when@test:
  avodel_framework:
    symfony_behat_api:
      doctrine: true
```

**Behat registration** (`behat.yml.dist`):
```yaml
default:
    suites:
        default:
            contexts:
                - Avodel\SymfonyBehatApi\Doctrine\DoctrineORMContext
```

### CLI Context

Provides step definitions for testing console commands.

**Configuration** (`config/packages/test/avodel_framework.yaml`):
```yaml
when@test:
  avodel_framework:
    symfony_behat_api:
      cli: true
```

**Behat registration** (`behat.yml.dist`):
```yaml
default:
    suites:
        default:
            contexts:
                - Avodel\SymfonyBehatApi\Cli\CliContext
```

### HTTP Client Context

Provides step definitions for mocking and testing HTTP client requests.

**Configuration** (`config/packages/test/avodel_framework.yaml`):
```yaml
when@test:
  avodel_framework:
    symfony_behat_api:
      http_client: true
```

**Behat registration** (`behat.yml.dist`):
```yaml
default:
    suites:
        default:
            contexts:
                - Avodel\SymfonyBehatApi\HttpClient\HttpClientContext
```

### JWT Context

Provides step definitions for testing JWT authentication flows.

**Configuration** (`config/packages/test/avodel_framework.yaml`):
```yaml
when@test:
  avodel_framework:
    symfony_behat_api:
      jwt: true
```

**Behat registration** (`behat.yml.dist`):
```yaml
default:
    suites:
        default:
            contexts:
                - Avodel\SymfonyBehatApi\Jwt\JwtContext
```

### Combined Example

Enable multiple contexts together:

```yaml
# config/packages/test/avodel_framework.yaml
when@test:
  avodel_framework:
    symfony_behat_api:
      rest_api: true
      doctrine: true
      cli: true
      http_client: true
      jwt: true
```

```yaml
# behat.yml.dist
default:
    suites:
        default:
            contexts:
                - Avodel\SymfonyBehatApi\RestApi\RestApiContext
                - Avodel\SymfonyBehatApi\Doctrine\DoctrineORMContext
                - Avodel\SymfonyBehatApi\Cli\CliContext
                - Avodel\SymfonyBehatApi\HttpClient\HttpClientContext
                - Avodel\SymfonyBehatApi\Jwt\JwtContext
```

## Installation

```bash
composer require avodel/framework-bundle
```

Register the bundle (if not using Symfony Flex):
```php
// config/bundles.php
return [
    Avodel\FrameworkBundle\AvodelFrameworkBundle::class => ['all' => true],
];
```

## Configuration Examples

**Minimal configuration:**
```yaml
avodel_framework:
    application_name: 'my-app'
```

**Full production configuration:**
```yaml
avodel_framework:
    application_name: '%env(APP_NAME)%'
    lock_db_migration: true
    healthcheck: true

    clock: ~

    uuid: ~

    logger:
        project_dir: '%kernel.project_dir%'
        elasticsearch:
            enabled: true
            url: '%env(ELASTICSEARCH_URL)%'
            username: '%env(ELASTICSEARCH_USER)%'
            password: '%env(ELASTICSEARCH_PASSWORD)%'
            index: '%env(APP_NAME)%-logs'
        exclusions:
            - { channel: request, message: 'Matched route "{route}".' }
            - { channel: security, message: 'Checking for authenticator support.' }

    jwt_http_client:
        clients:
            external_api:
                http_client: 'http_client.external_api'
                jwt_storage: 'cache.jwt_tokens'
                auth:
                    username: '%env(API_USERNAME)%'
                    password: '%env(API_PASSWORD)%'
```

**Test environment configuration:**
```yaml
avodel_framework:
    application_name: 'my-app'

    clock:
        behat: true

    uuid:
        behat: true

    symfony_behat_api:
        rest_api: true
        doctrine: true
        http_client: true
        cli: true
```
