# Healthcheck

A lightweight Symfony library package that provides a simple health check endpoint for applications. It enables load balancers, orchestrators (Kubernetes, Docker Swarm), and monitoring systems to verify that an application is running and responsive.

## Business Logic

### Core Concept

**Problem:** Modern distributed systems and container orchestrators need a reliable way to verify application availability. Without a dedicated health endpoint, monitoring tools must rely on checking the main application endpoints, which may have authentication, rate limiting, or complex response structures.

**Solution:** This package provides a minimal, stateless `/healthcheck` endpoint that:
- Returns immediately with a simple JSON response
- Has no authentication requirements
- Logs each request for debugging and audit purposes
- Can be integrated into any Symfony application with a single routing import

### Main Process

```
1. HTTP REQUEST
   │
   └── GET/POST /healthcheck
           │
           ▼
2. CONTROLLER INVOCATION
   │
   ├── Log debug message → "Healthcheck request received."
   │
   └── Generate response
           │
           ▼
3. HTTP RESPONSE
   │
   └── Return JSON: {"status": "ok"} (HTTP 200)
```

### Behavior Rules

| Scenario | Response | HTTP Code | Logged |
|----------|----------|-----------|--------|
| Endpoint called | `{"status": "ok"}` | 200 | Yes (DEBUG level) |
| Application down | No response (connection refused) | N/A | N/A |

**Design decisions:**
- The endpoint always returns "ok" if reachable — it does not perform deep health checks (database connectivity, external services, etc.)
- Uses PSR-3 logging interface for framework-agnostic logging
- Invokable controller pattern for minimal code footprint

## Architecture

```
src/
├── Controller/
│   └── HealthcheckController.php   # Single invokable controller
└── Resources/
    └── config/
        └── routing.yaml            # Route definition
```

See `src/Controller/HealthcheckController.php` for implementation details.

## API Endpoint

| Method | Endpoint | Description |
|--------|----------|-------------|
| GET/POST | `/healthcheck` | Returns application health status |

**Response:**
```json
{"status": "ok"}
```

## External Dependencies

| Package | Purpose |
|---------|---------|
| `psr/log` | PSR-3 Logger Interface for logging health check requests |
| `symfony/http-foundation` | HTTP response handling (JsonResponse) |

## Installation

Add the package to your project:

```bash
composer require avodel/healthcheck
```

Import routing in your Symfony application:

```yaml
# config/routes/healthcheck.yaml
app_healthcheck:
  resource: '../../vendor/avodel/healthcheck/src/Resources/config/routing.yaml'
```

## Configuration

No configuration required. The package uses:
- PSR-3 logger auto-injected by Symfony's service container
- Fixed route path `/healthcheck`

## Testing

Verify the endpoint is working:

```bash
curl http://localhost/healthcheck
# Expected: {"status":"ok"}
```
