# Avodel Logger

## Overview

A Monolog extension library for Symfony applications that provides enhanced logging capabilities for Elasticsearch/Kibana environments. The library solves the problem of structured logging by formatting log records in a Kibana-compatible JSON format, enriching logs with file/line information, processing exceptions with full context, and filtering out noisy log messages through configurable exclusions.

## Business Logic

### Core Concept

**Problem:** Standard Monolog logs lack the structured metadata required for effective Kibana dashboards and debugging. Exception stack traces are hard to analyze, log sources are unclear, and noisy logs clutter the output.

**Solution:** A set of Monolog processors, handlers, and formatters that:
1. Transform logs into Kibana-friendly JSON with timestamps and host info
2. Automatically capture source file and line for each log entry
3. Extract and format exception chains with context
4. Filter out configurable noisy messages before they reach handlers

### Main Process

```
LOG ENTRY
│
├── Channel (e.g., "app", "request", "doctrine")
├── Level (DEBUG, INFO, WARNING, ERROR, etc.)
├── Message
└── Context (arbitrary data, exceptions)
        │
        ▼
PROCESSORS (enrich the record)
│
├── FileLineProcessor
│   └── Adds 'file_line' from backtrace → "src/Controller/Api.php:42"
│
└── ExceptionProcessor
    ├── Extracts exception chain → file_lines[], messages[], codes[]
    ├── Merges ContextAwareException context into log context
    └── Downgrades 404/405 HTTP errors → WARNING level
        │
        ▼
EXCLUSIONS HANDLER (filter)
│
└── ExclusionsHandler
    └── Blocks records matching channel+message patterns → returns true to stop propagation
        │
        ▼
ELASTICSEARCH HANDLER (output)
│
├── ElasticSearchHandlerFactory
│   └── Returns real handler if enabled, NullHandler if disabled
│
└── KibanaFormatter
    └── Transforms record → JSON with @timestamp, host, level, context
```

### Algorithms and Rules

**Exception Processing:**
- Traverses the full exception chain via `getPrevious()`
- Limits nesting depth to 5 levels to prevent infinite loops
- HTTP exceptions (404, 405) are automatically downgraded to WARNING level
- Messages truncated to 200 characters for readability

**File/Line Detection:**
- Analyzes backtrace up to 8 frames deep
- Skips internal logger classes to find actual caller
- Converts absolute paths to project-relative paths

**Exclusion Matching:**
- Uses a pre-built hash map for O(1) lookup
- Matches on exact channel + exact message combination
- Returns `true` from handler to stop propagation

### Component Reference

| Component | Type | Purpose |
|-----------|------|---------|
| `KibanaFormatter` | Formatter | Transforms LogRecord to Kibana JSON |
| `ContextDataDumper` | Utility | Serializes complex context data for logging |
| `FileLineProcessor` | Processor | Adds source file:line to extra |
| `ExceptionProcessor` | Processor | Formats exceptions, extracts context |
| `ExclusionsHandler` | Handler | Filters out noisy log messages |
| `ElasticSearchHandlerFactory` | Factory | Conditionally creates ES handler |
| `ContextAwareExceptionInterface` | Interface | Exceptions that carry extra context |
| `ContextAwareTrait` | Trait | Default implementation for context-aware exceptions |

## Data Model

```
LogRecord (Monolog)
├── channel          - Log source identifier
├── level            - Severity (DEBUG → EMERGENCY)
├── message          - Log message
├── context          - User-provided data
│   ├── exception    - Throwable (processed by ExceptionProcessor)
│   └── [custom]     - Any additional context
└── extra            - Processor-added data
    ├── file_line    - Added by FileLineProcessor
    ├── exception_file_lines[]  - Exception locations
    └── exception_messages[]    - Exception messages

ContextAwareException (Application)
├── Throwable        - Standard PHP exception
└── context[]        - Additional debugging data merged into log
```

See `src/` for detailed implementation.

## External Dependencies

| Service | Purpose | API Type |
|---------|---------|----------|
| Elasticsearch | Log storage | HTTP via ElasticsearchLogstashHandler |
| Kibana | Log visualization | Consumes ES indices |


## Installation

```bash
composer require avodel/logger
```
