<?php

declare(strict_types=1);

namespace Avodel\Logger\Processor;

use Avodel\Logger\ContextAwareExceptionInterface;
use Monolog\Level;
use Monolog\LogRecord;
use Monolog\Processor\ProcessorInterface;
use Symfony\Component\HttpKernel\Exception\HttpException;
use Throwable;

final class ExceptionProcessor implements ProcessorInterface
{
    private const FILE_LINES_FIELD = 'exception_file_lines';
    private const MESSAGES_FIELD = 'exception_messages';
    private const CODE_FIELD = 'exception_codes';
    private const EXCEPTION_CLASS_FIELD = 'class';

    private const EXCEPTION_NESTING_LEVEL_LIMIT = 5;
    private const EXCLUDE_HTTP_CODES = [404, 405];

    public function __construct(private readonly string $projectDir)
    {
    }

    public function __invoke(LogRecord $record)
    {
        $recordData['context'] = $record->context;
        $level = $record->level;

        foreach ($record->context as $key => $value) {
            if ($value instanceof Throwable) {
                if ($key === 'exception' &&  $value instanceof HttpException && in_array($value->getStatusCode(), self::EXCLUDE_HTTP_CODES)) {
                    $level = Level::Warning;
                }

                $recordData = $this->handleExceptionContext($recordData, $value);
                $recordData['context'][$key] = $this->formatException($value);
            }
        }

        return $record->with(context: $recordData['context'], level: $level);
    }

    private function handleExceptionContext(array $record, Throwable $throwable): array
    {
        if ($throwable instanceof ContextAwareExceptionInterface) {
            foreach ($throwable->getContext() as $key => $value) {
                $record['context'][$key] = $value;
            }
        }

        while ($throwable !== null) {
            $record['context'][self::FILE_LINES_FIELD][] = $this->getFileLine($throwable);
            $record['context'][self::MESSAGES_FIELD][] = substr($throwable->getMessage(), 0, 200);
            $record['context'][self::CODE_FIELD][] = $throwable->getCode();

            // go to the next iteration
            $throwable = $throwable->getPrevious();
        }

        return $record;
    }

    private function formatException(Throwable $e, $level = 0)
    {
        if ($level > self::EXCEPTION_NESTING_LEVEL_LIMIT) {
            return 'Nesting limit reached';
        }

        $formatted = [
            self::EXCEPTION_CLASS_FIELD => get_class($e),
            'message' => substr($e->getMessage(), 0, 200),
            'code' => $e->getCode(),
            'file' => $this->getFileLine($e),
        ];

        foreach ($e->getTrace() as $i => $frame) {
            if (isset($frame['file'], $frame['line'])) {
                $formatted['trace']['#' . $i] = $frame['file'] . ':' . $frame['line'];
            }
        }

        $previous = $e->getPrevious();
        if (null !== $previous) {
            $formatted['previous'] = $this->formatException($previous, $level + 1);
        }

        return $formatted;
    }

    private function getFileLine(Throwable $t): string
    {
        $appRoot = $this->projectDir;
        $path = $t->getFile();
        $path = str_replace($appRoot . DIRECTORY_SEPARATOR, '', $path);

        return $path . ':' . $t->getLine();
    }
}
