<?php

declare(strict_types=1);

namespace Avodel\Logger\Processor;

use Monolog\Logger;
use Monolog\LogRecord;
use Monolog\Processor\ProcessorInterface;
use Psr\Log\AbstractLogger;
use ReflectionClass;

final class FileLineProcessor implements ProcessorInterface
{
    private const STACK_LIMIT = 8;

    private const SKIPPED_CLASSES = [self::class, Logger::class, AbstractLogger::class];

    private array $skippedFiles = [];

    public function __construct(private readonly string $projectDir)
    {
        foreach (self::SKIPPED_CLASSES as $skippedClass) {
            $reflector = new ReflectionClass($skippedClass);
            $this->skippedFiles[$reflector->getFileName()] = '';
        }
    }

    public function __invoke(LogRecord $record)
    {
        $backTraceFileLine = $this->getFileLineFromBackTrace();
        if (!$backTraceFileLine) {
            return $record;
        }

        $record->extra['file_line'] = $backTraceFileLine;

        return $record;
    }

    private function getFileLineFromBackTrace(): ?string
    {
        $throwPlace = $this->findThrowPlace(debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, self::STACK_LIMIT));
        if ($throwPlace === null) {
            return null;
        }

        return $this->formatFileLineArray($throwPlace);
    }

    private function findThrowPlace(array $debugBacktrace): ?array
    {
        foreach ($debugBacktrace as $tracePart) {
            if ($this->isSkippedTracePart($tracePart)) {
                continue;
            }

            return $tracePart;
        }

        return null;
    }

    private function isSkippedTracePart(array $tracePart): bool
    {
        if (!isset($tracePart['file'])) {
            return true;
        }

        return array_key_exists($tracePart['file'], $this->skippedFiles);
    }

    private function formatFileLineArray(array $throwPlace): string
    {
        $relatedFilePath = str_replace($this->projectDir, '', $throwPlace['file']);

        return $relatedFilePath . ':' . $throwPlace['line'];
    }
}
