<?php

declare(strict_types=1);

namespace Avodel\Logger\Kibana;

use Symfony\Component\VarDumper\Cloner\VarCloner;
use Symfony\Component\VarDumper\Dumper\CliDumper;
use Throwable;
use function array_map;
use function implode;
use function is_array;
use function is_scalar;
use function sprintf;
use function str_repeat;
use function substr;
use function var_export;
use const PHP_EOL;

final class ContextDataDumper
{
    private const ARRAY_DEPTH_LIMIT = 5;
    private const INDENT_SIZE = 2;

    public static function dump($value, $level = 1): string
    {
        if ($level > self::ARRAY_DEPTH_LIMIT) {
            return sprintf('LOGGING DEPTH LIMIT OF %d IS REACHED', self::ARRAY_DEPTH_LIMIT);
        }

        if (is_scalar($value)) {
            return var_export($value, true);
        }

        if (null === $value) {
            return 'null';
        }

        if (is_array($value)) {
            if (self::isCompact($value)) {
                if (0 === $level && 0 === count($value)) {
                    return '';
                }

                return self::serializeCompactArray($value, $level);
            }

            return self::serializeArray($value, $level);
        }

        return self::vardump($value, $level);
    }

    private static function isCompact(array $array): bool
    {
        $i = 0;
        foreach ($array as $key => $value) {
            if ($i !== $key) {
                return false;
            }

            // Representing such values requires non-compact output
            if (!is_scalar($value) && null !== $value) {
                return false;
            }

            ++$i;
        }

        return true;
    }

    private static function serializeArray(array $value, int $level): string
    {
        $result = '';
        if ($level > 0) {
            $result .= '[' . PHP_EOL;
        }

        foreach ($value as $k => $v) {
            if ($level > 0) {
                $result .= str_repeat(' ', $level * self::INDENT_SIZE);
            }
            $result .= self::dump($k) . ' => ' . self::dump($v, $level + 1) . ',' . PHP_EOL;
        }

        if ($level > 0) {
            $result .= str_repeat(' ', ($level - 1) * self::INDENT_SIZE);
            $result .= ']';
        } else {
            $result = substr($result, 0, -1);
        }

        return $result;
    }

    private static function serializeCompactArray(array $value, int $level): string
    {
        $values = array_map(static function ($v) use ($level) {
            return self::dump($v, $level + 1);
        }, $value);

        return '[' . implode(', ', $values) . ']';
    }

    private static function vardump($object, int $level): string
    {
        $cloner = new VarCloner();
        $cloner->setMaxItems(50);
        $dumper = new CliDumper();
        $output = '';

        try {
            $objectToDump = $cloner
                ->cloneVar($object)
                ->withMaxItemsPerDepth(50)
                ->withRefHandles(false);
        } catch (Throwable $e) {
            return 'Error when dumping object: ' . $e->getMessage();
        }

        $dumper->dump(
            $objectToDump,
            function ($line, $depth) use (&$output, $level): void {
                // A negative depth means "end of dump"
                if ($depth >= 0) {
                    $output .= str_repeat(' ', ($depth + $level - 1) * self::INDENT_SIZE) . $line . "\n";
                }
            }
        );

        // Trims the last newline
        return substr($output, 0, -1);
    }
}
