# Migration Lock

A Symfony library that prevents concurrent execution of Doctrine migrations. In distributed environments (multiple servers, containers, or CI/CD pipelines), simultaneous migration attempts can cause database corruption or conflicts. This library uses Symfony's Lock component to ensure only one migration process runs at a time, with automatic waiting and deadlock prevention.

## Business Logic

### Core Concept

**Problem**: When multiple application instances start simultaneously (e.g., during deployment), each may attempt to run `doctrine:migrations:migrate`. This causes race conditions, duplicate migrations, or partial database states.

**Solution**: Acquire a distributed lock before migration execution. Other processes wait for the lock to be released or timeout safely.

### Main Process

```
1. CONSOLE COMMAND START
   │
   ├── Command: doctrine:migrations:migrate
   └── Event: ConsoleEvents::COMMAND
           │
           ▼
2. LOCK ACQUISITION
   │
   ├── Create lock key: {prefix}:doctrine:migrations:migrate
   ├── TTL: 300 seconds (deadlock prevention)
   └── acquire(blocking=true) → Wait if lock exists
           │
           ▼
3. MIGRATION EXECUTION
   │
   └── Doctrine runs migrations (normal flow)
           │
           ▼
4. CONSOLE COMMAND TERMINATE
   │
   ├── Event: ConsoleEvents::TERMINATE
   └── Release lock → Next waiting process proceeds
```

### Lock Behavior

| Scenario | Behavior |
|----------|----------|
| No lock exists | Acquire immediately, proceed |
| Lock held by another process | Wait (blocking) until released |
| Process crashes during migration | Lock auto-expires after 300s (TTL) |
| Migration completes | Lock released immediately |

### Key Parameters

| Parameter | Value | Purpose |
|-----------|-------|---------|
| Target command | `doctrine:migrations:migrate` | Only this command is locked |
| Lock TTL | 300 seconds | Prevents infinite deadlock on crash |
| Blocking mode | `true` | Processes wait rather than fail |
| Lock key format | `{prefix}:command` | Allows multi-app isolation |

## Architecture

```
LockDoctrineMigrationListener (EventSubscriberInterface)
├── onCommand()    - Acquires lock on migration start
├── onTerminate()  - Releases lock on migration end
└── Dependencies:
    ├── LockFactory  - Creates distributed locks (Redis, Semaphore, etc.)
    └── prefix       - Application identifier for lock key
```

See `src/LockDoctrineMigrationListener.php` for implementation details.

## External Dependencies

| Component | Purpose |
|-----------|---------|
| `symfony/lock` | Distributed locking mechanism |
| `symfony/console` | Console event hooks |
| `symfony/event-dispatcher` | Event subscription |
| Lock Store (user-provided) | Persistence backend (Redis, Semaphore, Flock, etc.) |

## Installation

```bash
composer require avodel/migration-lock
```

## How It Works (Technical Detail)

1. **Event Subscription**: Listener subscribes to `ConsoleEvents::COMMAND` and `ConsoleEvents::TERMINATE`
2. **Command Detection**: Only `doctrine:migrations:migrate` triggers locking
3. **Lock Creation**: Uses `LockFactory::createLock()` with configured prefix and TTL
4. **Blocking Acquire**: `acquire(true)` blocks until lock is available
5. **Auto-Cleanup**: Lock released on command termination; TTL handles crashes
