<?php

namespace Avodel\SwaggerCodeGenerator\Generator;

use Avodel\SwaggerCodeGenerator\Generator\Model\ControllerModel;
use Avodel\SwaggerCodeGenerator\Generator\Model\Endpoint;
use Avodel\SwaggerCodeGenerator\Generator\Model\EndpointParameter;
use Avodel\SwaggerCodeGenerator\Generator\Model\QueryParameter;
use Avodel\SwaggerCodeGenerator\Generator\Util\FileManager;
use Avodel\SwaggerCodeGenerator\Generator\Util\PropertyTypeResolver;
use Avodel\SwaggerCodeGenerator\Parser\Schema\Schema;
use Avodel\SwaggerCodeGenerator\Parser\Schema\Specification;

class ControllerGenerator
{
    public function __construct(
        private readonly FileManager $fileManager,
        private readonly PropertyTypeResolver $propertyTypeResolver,
        private readonly SwaggerAnnotationsGenerator $swaggerAnnotationsGenerator,
        private readonly ExceptionGenerator $exceptionGenerator,
    )
    {
    }

    public function generate(
        Specification $specification,
        string $outputDir,
        string $namespaceValue,
        string $clientName,
        string $modelNamespace,
        string $exceptionOutputDir,
        string $exceptionNamespace,
    ): void
    {
        $controller = new ControllerModel();
        $controller->namespace = $namespaceValue;
        $controller->name = 'Abstract' . $clientName . 'ApiController';

        $useStatements = [];

        foreach ($specification->getSchemasCollection()->getAll() as $ref => $schema) {
            $modelName = $schema::getModelNameFromRef($ref);
            $useStatements[] = $modelNamespace . '\\' . $modelName;
        }

        foreach ($specification->getPathsCollection()->getAll() as $route => $path) {
            foreach ($path->getOperationsCollection()->getAll() as $requestMethod => $operationSpec) {
                $endpoint = new Endpoint();
                $controller->endpoints[] = $endpoint;
                $endpoint->name = $operationSpec->getOperationId();
                $endpoint->path = $route;
                $endpoint->requestMethod = $requestMethod;
                $endpoint->swaggerAnnotations = $this->swaggerAnnotationsGenerator->generate($operationSpec);

                foreach ($operationSpec->getParametersCollection()->getAll() as $paramName => $paramSpec) {
                    if ($paramSpec->getIn() === 'query') {
                        $queryParameter = new QueryParameter();
                        $queryParameter->name = $paramName;
                        $queryParameter->type = $this->propertyTypeResolver->resolve($paramSpec->getType() ?? 'mixed', $paramSpec->getFormat());
                        $queryParameter->nullable = !$paramSpec->isRequired();
                        $endpoint->queryParameters[] = $queryParameter;

                        continue;
                    }
                    $param = new EndpointParameter();
                    $param->name = $paramName;
                    $param->useMapRequestPayload = $paramSpec->getIn() === 'body';

                    if ($paramSpec->getSchema() !== null) {
                        $paramModelClass = Schema::getModelNameFromRef($paramSpec->getSchema()->getRef());
                        $useStatements[] = $modelNamespace . '\\' . $paramModelClass;
                        $param->type = $paramModelClass;
                    } else {
                        $param->type = $this->propertyTypeResolver->resolve($paramSpec->getType() ?? 'mixed', $paramSpec->getFormat());
                    }

                    $param->nullable = !$paramSpec->isRequired();

                    $endpoint->parameters[] = $param;
                }

                foreach ($operationSpec->getResponsesCollection()->getAll() as $httpStatusCode => $responseSpec) {
                    if ($httpStatusCode >= 400) {
                        $exceptionClass = $this->exceptionGenerator->generateServer(
                            $responseSpec->getDescription(),
                            $httpStatusCode,
                            $exceptionOutputDir,
                            $exceptionNamespace
                        );
                        $useStatements[] = $exceptionClass;
                        $baseExceptionClassName = basename(str_replace('\\', '/', $exceptionClass));
                        $endpoint->phpDocs[] = '@throws ' . $baseExceptionClassName;
                    }
                }
            }
        }

        $controller->uses = array_values(array_unique($useStatements));

        $code = $this->fileManager->parseTemplate('controller.tpl.php', ['controller' => $controller]);
        $this->fileManager->dumpFile("$outputDir/{$controller->name}.php", $code);
    }
}