<?php

namespace Avodel\SwaggerCodeGenerator\Generator;

use Avodel\SwaggerCodeGenerator\Generator\Model\ClassModel;
use Avodel\SwaggerCodeGenerator\Generator\Model\ClassProperty;
use Avodel\SwaggerCodeGenerator\Generator\Util\FileManager;
use Avodel\SwaggerCodeGenerator\Generator\Util\PropertyTypeResolver;
use Avodel\SwaggerCodeGenerator\Parser\Schema\Property;
use Avodel\SwaggerCodeGenerator\Parser\Schema\Schema;
use Avodel\SwaggerCodeGenerator\Parser\Schema\Specification;

class ModelGenerator
{
    public function __construct(
        private readonly PropertyTypeResolver $propertyTypeResolver,
        private readonly FileManager $fileManager,
    )
    {
    }

    public function generate(Specification $specification, string $outputDir, string $namespaceValue): void
    {
        foreach ($specification->getSchemasCollection()->getAll() as $ref => $schema) {
            if ($schema->getEnum()) {
                continue;
            }

            $class = new ClassModel();
            $class->namespace = $namespaceValue;

            $modelName = $schema::getModelNameFromRef($ref);
            $class->name = $modelName;
            $class->namespace = $namespaceValue;
            $class->builderName = $modelName.'Builder';

            $useStatements = [];

            foreach ($schema->getPropertiesCollection()->getAll() as $propertyName => $propertySpec) {
                $classProperty = new ClassProperty();
                $classProperty->name = $propertyName;
                $classProperty->nullable = !in_array($propertyName, $schema->getRequiredProperties(), true);
                $class->properties[] = $classProperty;

                if ($propertySpec->getRef()) {
                    $classProperty->type = Schema::getModelNameFromRef($propertySpec->getRef());

                    continue;
                }

                if ($propertySpec->getType() === 'string' && $propertySpec->getFormat() === 'date-time') {
                    $useStatements[] = 'Symfony\\Component\\Serializer\\Annotation\\Context';
                    $useStatements[] = 'Symfony\\Component\Serializer\\Normalizer\\DateTimeNormalizer';
                    $classProperty->attributes[] = '#[Context([DateTimeNormalizer::FORMAT_KEY => \'Y-m-d\TH:i:s\Z\'])]';
                }

                if ($propertySpec->getType() === 'string' && $propertySpec->getFormat() === 'date') {
                    $useStatements[] = 'Symfony\\Component\\Serializer\\Annotation\\Context';
                    $useStatements[] = 'Symfony\\Component\Serializer\\Normalizer\\DateTimeNormalizer';
                    $classProperty->attributes[] = '#[Context([DateTimeNormalizer::FORMAT_KEY => \'Y-m-d\'])]';
                }

                $propertyType = $propertySpec->getType() ?? 'mixed';
                $classProperty->type = $this->propertyTypeResolver->resolve($propertyType, $propertySpec->getFormat());

                if ($propertyType === 'array') {
                    $arrayType = $this->resolveArrayType($propertySpec);
                    $classProperty->genericType = $arrayType;
                }
            }

            $class->uses = array_values(array_unique($useStatements));

            $code = $this->fileManager->parseTemplate('model.tpl.php', ['class' => $class]);
            $this->fileManager->dumpFile("$outputDir/$modelName.php", $code);

            $code = $this->fileManager->parseTemplate('model_builder.tpl.php', ['class' => $class]);
            $this->fileManager->dumpFile("$outputDir/{$class->builderName}.php", $code);
        }
    }

    private function resolveArrayType(Property $propertySpec): string
    {
        if ($propertySpec->getItems()->getRef()) {
            return Schema::getModelNameFromRef($propertySpec->getItems()->getRef());
        }

        if ($propertySpec->getItems()->getType()) {
            return $propertySpec->getItems()->getType();
        }

        return 'mixed';
    }
}