<?php

namespace Avodel\SwaggerCodeGenerator\Parser;

use Avodel\SwaggerCodeGenerator\Parser\Schema\Operation;
use Avodel\SwaggerCodeGenerator\Parser\Schema\OperationsCollection;
use Avodel\SwaggerCodeGenerator\Parser\Schema\Parameter;
use Avodel\SwaggerCodeGenerator\Parser\Schema\ParametersCollection;
use Avodel\SwaggerCodeGenerator\Parser\Schema\Path;
use Avodel\SwaggerCodeGenerator\Parser\Schema\PathsCollection;
use Avodel\SwaggerCodeGenerator\Parser\Schema\Property;
use Avodel\SwaggerCodeGenerator\Parser\Schema\PropertyCollection;
use Avodel\SwaggerCodeGenerator\Parser\Schema\PropertyItems;
use Avodel\SwaggerCodeGenerator\Parser\Schema\Response;
use Avodel\SwaggerCodeGenerator\Parser\Schema\ResponsesCollection;
use Avodel\SwaggerCodeGenerator\Parser\Schema\Schema;
use Avodel\SwaggerCodeGenerator\Parser\Schema\SchemaItems;
use Avodel\SwaggerCodeGenerator\Parser\Schema\SchemasCollection;
use Avodel\SwaggerCodeGenerator\Parser\Schema\Specification;
use Symfony\Component\Yaml\Yaml;

class Parser
{
    public function parse(string $yamlFile): Specification
    {
        if (!file_exists($yamlFile)) {
            throw new \RuntimeException('File not found: ' . $yamlFile);
        }

        $yamlContents = file_get_contents($yamlFile);
        $openapi = Yaml::parse($yamlContents);
        $schemes = $this->parseSchemas($openapi);
        $paths = $this->parsePaths($openapi);

        return new Specification($schemes, $paths);
    }

    private function parseSchemas(array $swagger): SchemasCollection
    {
        $schemaCollection = new SchemasCollection();

        foreach ($swagger['definitions'] ?? [] as $schemaName => $schema) {
            $parsedSchema = $this->parseSchema($schema);
            $schemaCollection->addSchema('#/definitions/' . $schemaName, $parsedSchema);
        }

        return $schemaCollection;
    }

    private function parseSchema(array $schema): ?Schema
    {
        if (!$schema) {
            return null;
        }

        $properties = $this->parseProperties($schema['properties'] ?? []);
        $items = null;

        if (!empty($schema['items'])) {
            $items = new SchemaItems(
                $schema['items']['$ref'] ?? null,
            );
        }

        return new Schema(
            $schema['type'] ?? null,
            $properties,
            $schema['required'] ?? [],
            $schema['$ref'] ?? null,
            $items,
            $schema['enum'] ?? []
        );
    }

    private function parseProperties(array $schemaProperties): PropertyCollection
    {
        $propertyCollection = new PropertyCollection();

        foreach ($schemaProperties as $propertyName => $schemaProperty) {
            $items = null;

            if (!empty($schemaProperty['items']['type']) || !empty($schemaProperty['items']['$ref'])) {
                $items = new PropertyItems(
                    $schemaProperty['items']['type'] ?? null,
                    $schemaProperty['items']['$ref'] ?? null
                );
            }

            $property = new Property(
                $schemaProperty['type'] ?? null,
                $schemaProperty['format'] ?? null,
                $schemaProperty['description'] ?? null,
                $items,
                $schemaProperty['$ref'] ?? null
            );

            $propertyCollection->addProperty($propertyName, $property);
        }

        return $propertyCollection;
    }

    private function parsePaths(array $openapi): PathsCollection
    {
        $pathsCollection = new PathsCollection();

        foreach ($openapi['paths'] as $path => $operations) {
            $operations = $this->parseOperations($operations);

            $pathData = new Path($operations);
            $pathsCollection->addPath($path, $pathData);
        }

        return $pathsCollection;
    }

    private function parseOperations(array $operations): OperationsCollection
    {
        $operationsCollection = new OperationsCollection();

        foreach ($operations as $method => $operationArray) {
            $parameters = $this->parseParameters($operationArray['parameters'] ?? []);
            $responsesCollection = $this->parseResponses($operationArray);

            $operation = new Operation(
                $operationArray['summary'] ?? null,
                $operationArray['description'] ?? null,
                $operationArray['operationId'],
                $parameters,
                $responsesCollection,
                $operationArray['tags'] ?? []
            );
            $operationsCollection->addOperation($method, $operation);
        }

        return $operationsCollection;
    }

    private function parseParameters(array $parametersArray): ParametersCollection
    {
        $parameters = new ParametersCollection();

        foreach ($parametersArray as $parameter) {
            $parsedParameter = new Parameter(
                $parameter['in'],
                $parameter['required'] ?? false,
                $parameter['description'] ?? null,
                $parameter['type'] ?? null,
                $parameter['format'] ?? null,
                $this->parseSchema($parameter['schema'] ?? [])
            );

            $parameters->addParameter($parameter['name'], $parsedParameter);
        }

        return $parameters;
    }

    private function parseResponses(array $responses): ResponsesCollection
    {
        $responsesCollection = new ResponsesCollection();

        foreach ($responses['responses'] as $statusCode => $response) {
            $parsedResponse = new Response(
                $response['description'] ?? null,
                $this->parseSchema($response['schema'] ?? []),
            );
            $responsesCollection->addResponse($statusCode, $parsedResponse);
        }

        return $responsesCollection;
    }
}
