<?php

namespace Avodel\SwaggerCodeGenerator\Generator;

use Avodel\SwaggerCodeGenerator\Generator\Model\ClientApi;
use Avodel\SwaggerCodeGenerator\Generator\Model\Operation;
use Avodel\SwaggerCodeGenerator\Generator\Model\Param;
use Avodel\SwaggerCodeGenerator\Generator\Model\StatusCode;
use Avodel\SwaggerCodeGenerator\Generator\Util\FileManager;
use Avodel\SwaggerCodeGenerator\Generator\Util\PropertyTypeResolver;
use Avodel\SwaggerCodeGenerator\Generator\Util\UnderscoreToCamelcaseConverter;
use Avodel\SwaggerCodeGenerator\Parser\Schema\Schema;
use Avodel\SwaggerCodeGenerator\Parser\Schema\Specification;

final class ClientGenerator
{
    public function __construct(
        private readonly FileManager $fileManager,
        private readonly PropertyTypeResolver $propertyTypeResolver,
        private readonly ExceptionGenerator $exceptionGenerator,
        private readonly ClientServiceDefinitionsRegistrar $serviceDefinitionsRegistrar,
    )
    {
    }

    public function generate(
        Specification $specification,
        string $outputDir,
        string $namespaceValue,
        string $clientName,
        string $modelNamespace,
        string $exceptionOutputDir,
        string $exceptionNamespace,
        string $serviceDefinitionsOutputDir,
    ): void {
        $clientApi = new ClientApi();
        $clientApi->namespace = $namespaceValue;
        $clientApi->name = 'Rest' . $clientName . 'Api';
        $clientApi->interfaceName = $clientName . 'ApiInterface';

        $useStatements = [];

        foreach ($specification->getPathsCollection()->getAll() as $route => $path) {
            $updatedRoute = $route;

            foreach ($path->getOperationsCollection()->getAll() as $requestMethod => $operationSpec) {
                $operation = new Operation();
                $operation->name = $operationSpec->getOperationId();
                $operation->returnTypes = [];
                $clientApi->operations[] = $operation;
                $operation->requestMethod = strtoupper($requestMethod);

                foreach ($operationSpec->getResponsesCollection()->getAll() as $code => $responseSpec) {
                    $statusCode = new StatusCode();
                    $statusCode->code = $code;
                    $operation->statusCodes[$code] = $statusCode;
                    $responseSpecSchema = $responseSpec->getSchema();

                    if ($responseSpecSchema) {
                        if ($responseSpecSchema->getRef() !== null) {
                            $dataClass = Schema::getModelNameFromRef($responseSpec->getSchema()->getRef());
                            $useStatements[] = $modelNamespace . '\\' . $dataClass;
                            $operation->returnTypes[] = $dataClass;
                            $statusCode->dataClass = $dataClass;
                        } elseif ($responseSpecSchema->getType() === 'array') {
                            $dataClass = Schema::getModelNameFromRef($responseSpecSchema->getSchemaItems()?->getRef());
                            $useStatements[] = $modelNamespace . '\\' . $dataClass;
                            $operation->returnTypes[] = 'array';
                            $statusCode->dataClass = $dataClass;
                            $statusCode->isArray = true;

                            $operation->returnTypehints[] = "array<$dataClass>";
                        } else {
                            $operation->returnTypes[] = $responseSpecSchema->getType();
                        }
                    }

                    if ($code >= 400) {
                        $exceptionClass = $this->exceptionGenerator->generateClient($responseSpec->getDescription(), $exceptionOutputDir, $exceptionNamespace);
                        $useStatements[] = $exceptionClass;
                        $baseExceptionClassName = basename(str_replace('\\', '/', $exceptionClass));
                        $operation->throwsExceptions[] = $baseExceptionClassName;
                        $statusCode->exception = $baseExceptionClassName;
                    }
                }

                foreach ($operationSpec->getParametersCollection()->getAll() as $paramName => $paramSpec) {
                    $param = new Param();
                    $param->name = $paramName;
                    $param->camelCaseName = UnderscoreToCamelcaseConverter::convert($paramName);
                    $param->in = $paramSpec->getIn();

                    if ($paramSpec->getSchema() !== null) {
                        $paramType = Schema::getModelNameFromRef($paramSpec->getSchema()->getRef());
                        $useStatements[] = $modelNamespace . '\\' . $paramType;
                        $param->type = $paramType;
                    } else {
                        $param->type = $this->propertyTypeResolver->resolve($paramSpec->getType() ?? 'mixed', $paramSpec->getFormat());
                    }

                    $param->nullable = !$paramSpec->isRequired();
                    $param->format = $paramSpec->getFormat();

                    switch ($paramSpec->getIn()) {
                        case 'query':
                            $operation->queryParams[] = $param;
                            break;
                        case 'path':
                            $operation->pathParams[] = $param;
                            $updatedRoute = str_replace(sprintf('{%s}', $paramName), '$' . $paramName , $updatedRoute);
                            break;
                        case 'body':
                            $operation->bodyParam = $param;
                            break;
                        default:
                            throw new \RuntimeException('Unsupported param "in" value.');
                    }

                    $operation->params[] = $param;
                }

                $operation->uri = $updatedRoute;
            }
        }

        $clientApi->uses = array_values(array_unique($useStatements));

        $clientInterfaceCode = $this->fileManager->parseTemplate('client_interface.tpl.php', ['clientApi' => $clientApi]);
        $clientClass = $this->fileManager->parseTemplate('client.tpl.php', ['clientApi' => $clientApi]);
        $this->fileManager->dumpFile("$outputDir/{$clientApi->name}.php", $clientClass);
        $this->fileManager->dumpFile("$outputDir/{$clientApi->interfaceName}.php", $clientInterfaceCode);
        $this->serviceDefinitionsRegistrar->registry($serviceDefinitionsOutputDir, $clientName, $clientApi->namespace . '\\' . $clientApi->name, $clientApi->namespace . '\\' . $clientApi->interfaceName);
    }
}
