# Swagger Code Generator

A PHP library that generates type-safe PHP code from OpenAPI/Swagger 2.0 specifications. It transforms API contracts into production-ready Models, REST API Clients, and Abstract Controllers with full Symfony integration support.

The library solves the problem of maintaining consistency between API specifications and PHP implementation by automating code generation, eliminating manual DTO creation, and ensuring type safety across client-server boundaries.

## Business Logic

### Core Concept

**Problem:** Manually writing DTOs, API clients, and controllers from OpenAPI specs is error-prone, time-consuming, and leads to drift between specification and implementation.

**Solution:** Parse OpenAPI YAML → Transform to internal models → Generate PHP code using templates.

### Main Process

```
1. CLI INVOCATION
   │
   ├── --spec=path/to/spec.yaml     (required)
   ├── --feature=FeatureName        (required)
   ├── --output-dir=.generated      (optional)
   ├── --with-client                (optional)
   └── --with-server                (optional)

2. PARSING (Parser::parse)
   │
   ├── Read YAML file
   ├── Extract definitions → SchemasCollection
   │   ├── Schema objects (type, properties, required fields)
   │   └── Property objects (type, format, pattern, nullable)
   └── Extract paths → PathsCollection
       ├── Path objects (endpoint URL)
       └── Operation objects (method, parameters, responses)

3. CODE GENERATION (Generator::generate)
   │
   ├── ModelGenerator
   │   ├── Create immutable PHP classes from schemas
   │   ├── Add Symfony Serializer attributes
   │   ├── Add Validator constraints (Regex, Valid)
   │   └── Generate Builder classes
   │
   ├── EnumGenerator
   │   └── Create PHP 8.1+ enum classes from enum schemas
   │
   ├── ClientGenerator (if --with-client)
   │   ├── Generate REST client implementation
   │   ├── Generate client interface
   │   ├── Generate HTTP exception classes
   │   └── Register services in rest_api_clients.yaml
   │
   └── ControllerGenerator (if --with-server)
       ├── Generate abstract controller extending AbstractController
       ├── Add #[Route] and #[MapRequestPayload] attributes
       ├── Generate OpenAPI annotations
       └── Generate server exception classes

4. OUTPUT
   │
   └── Type-safe PHP classes in <output-dir>/<Feature>/
       ├── Model/          → DTOs + Builders
       ├── Client/         → API client + interface + exceptions
       └── Server/         → Abstract controller + exceptions
```

### Type Resolution Rules

| OpenAPI Type | OpenAPI Format | PHP Type |
|--------------|----------------|----------|
| string | - | string |
| string | date | \DateTimeInterface |
| string | date-time | \DateTimeInterface |
| integer | int32/int64 | int |
| number | float/double | float |
| boolean | - | bool |
| array | - | array |
| object/$ref | - | ClassName |

### Property Transformation

| OpenAPI Property | Generated PHP |
|------------------|---------------|
| `snake_case` name | `#[SerializedName('snake_case')]` + camelCase property |
| `required: [field]` | Non-nullable property |
| `pattern: "regex"` | `#[Regex(pattern: '/regex/')]` |
| `format: date` | `#[Context([DateTimeNormalizer::FORMAT_KEY => 'Y-m-d'])]` |
| `$ref` to object | `#[Valid]` constraint |

## Data Model

```
Generator (orchestrator)
├── Parser
│   └── Specification              - parsed OpenAPI document
│       ├── SchemasCollection      - all schema definitions
│       │   └── Schema             - single schema (object/enum)
│       │       └── PropertyCollection
│       │           └── Property   - field with type, format, pattern
│       └── PathsCollection        - all API paths
│           └── Path               - single endpoint
│               └── OperationsCollection
│                   └── Operation  - HTTP method handler
│                       ├── ParametersCollection
│                       │   └── Parameter - path/query/body param
│                       └── ResponsesCollection
│                           └── Response - status code + schema
│
├── ModelGenerator → ClassModel, ClassProperty
├── EnumGenerator → EnumClass
├── ClientGenerator → ClientApi, Operation, Param, ExceptionModel
└── ControllerGenerator → ControllerModel, Endpoint, EndpointParameter
```

See `src/Parser/Schema/` for parsed spec objects and `src/Generator/Model/` for generation DTOs.

## External Dependencies

| Package | Purpose |
|---------|---------|
| symfony/yaml | Parse OpenAPI YAML specifications |
| symfony/filesystem | Directory creation and file writing |

**Generated code requires:**
| Package | Usage in Generated Code |
|---------|------------------------|
| symfony/serializer | JSON serialization/deserialization |
| symfony/http-client | HTTP requests in API clients |
| symfony/validator | Property validation constraints |
| nelmio/api-doc-bundle | OpenAPI annotations on controllers |

## CLI Usage

```bash
php bin/swagger-gen \
    --spec=path/to/openapi.yaml \
    --feature=Product \
    --output-dir=.generated \
    --with-client \
    --with-server
```

| Option | Required | Description |
|--------|----------|-------------|
| `--spec` | Yes | Path to OpenAPI/Swagger YAML file |
| `--feature` | Yes | Feature name (becomes namespace: `Generated\<Feature>`) |
| `--output-dir` | No | Output directory (default: `.generated`) |
| `--with-client` | No | Generate REST client + interface |
| `--with-server` | No | Generate abstract controller |

## Generated Output Structure

```
<output-dir>/
├── <Feature>/
│   ├── Model/
│   │   ├── <Schema>.php           # Immutable DTO
│   │   ├── <Schema>Builder.php    # Builder pattern
│   │   └── <EnumSchema>.php       # PHP 8.1+ enum
│   ├── Client/
│   │   ├── Rest<Feature>Api.php   # HTTP client implementation
│   │   ├── <Feature>ApiInterface.php
│   │   └── Exception/
│   │       └── <ErrorName>Exception.php
│   └── Server/
│       ├── Abstract<Feature>ApiController.php
│       └── Exception/
│           └── <ErrorName>Exception.php
└── rest_api_clients.yaml          # Symfony service definitions
```

## Installation

```bash
composer require avodel/swagger-code-generator
```

### Symfony Integration

Add to `composer.json` for automatic generation on install/update:

```json
{
  "scripts": {
    "auto-scripts": {
      "./vendor/avodel/swagger-code-generator/bin/swagger-gen --spec=./spec.yaml --feature=MyApi --with-client --output-dir=.generated": "php-script"
    },
    "post-install-cmd": ["@auto-scripts"],
    "post-update-cmd": ["@auto-scripts"]
  }
}
```

Import generated client services:

```yaml
# config/services.yaml
imports:
    - { resource: '../.generated/rest_api_clients.yaml' }
```

## Example OpenAPI Spec

```yaml
swagger: '2.0'
info:
  title: Pet Store
  version: 1.0.0
paths:
  /pets/{petId}:
    get:
      operationId: getPetById
      parameters:
        - name: petId
          in: path
          required: true
          type: integer
      responses:
        200:
          schema:
            $ref: '#/definitions/Pet'
        404:
          description: Pet not found
definitions:
  Pet:
    type: object
    required: [id, name]
    properties:
      id:
        type: integer
      name:
        type: string
      birth_date:
        type: string
        format: date
      email:
        type: string
        pattern: '^[\w-\.]+@([\w-]+\.)+[\w-]{2,4}$'
```

**Generated Model:**
```php
final class Pet {
    public function __construct(
        private readonly int $id,
        private readonly string $name,
        #[SerializedName('birth_date')]
        #[Context([DateTimeNormalizer::FORMAT_KEY => 'Y-m-d'])]
        private readonly ?\DateTimeInterface $birthDate = null,
        #[Regex(pattern: '/^[\w-\.]+@([\w-]+\.)+[\w-]{2,4}$/')]
        private readonly ?string $email = null,
    ) {}
}
```

**Generated Client:**
```php
interface PetApiInterface {
    public function getPetById(int $petId): Pet;
}
```
