# Symfony Behat API

## Overview

A Behat extension package providing reusable context classes for testing REST APIs in Symfony applications. It abstracts common testing patterns into Gherkin steps for HTTP requests/responses, JWT authentication, external HTTP client mocking, database isolation, and CLI command execution.

The package eliminates repetitive test code by offering pre-built, composable Behat contexts that integrate directly with Symfony's kernel, enabling behavior-driven API testing without boilerplate.

## Business Logic

### Core Concept

The extension bridges Behat's scenario execution and Symfony's HTTP kernel. When a Gherkin step like `When I request "GET /api/users"` executes, the framework translates it into an actual HTTP request processed by the Symfony application. State (responses, tokens, HTTP stubs) is maintained across steps within a scenario.

### Main Process

```
BEHAT SCENARIO EXECUTION
│
├── 1. SETUP PHASE (Given steps)
│   ├── Configure request payload/headers
│   ├── Set JWT authentication token
│   ├── Define HTTP client stubs for external APIs
│   └── Database is automatically purged (@BeforeScenario)
│
├── 2. ACTION PHASE (When steps)
│   ├── Execute HTTP request via Symfony kernel
│   │   ├── Apply stored headers and Content-Type
│   │   ├── Inject Bearer token if present
│   │   └── Kernel.handle(Request) → Response
│   └── Execute CLI command if needed
│
├── 3. ASSERTION PHASE (Then steps)
│   ├── Verify response status code
│   ├── Compare JSON/text response body
│   └── Check HTTP interactions were consumed
│
└── 4. CLEANUP (@AfterStep)
    └── Verify all stubbed HTTP interactions were used
```

### Key Algorithms

**Request Execution:**
1. Build server array with headers and `Content-Type: application/json`
2. If `HttpAuthorizationTokenProvider` has token, add `Authorization: Bearer <token>`
3. Create Symfony Request and pass through `kernel.handle()`
4. Store Response for subsequent assertions

**HTTP Interaction Matching:**
- Stubs are indexed by: `method → URL → content-type → request body`
- JSON bodies are compared semantically (not by string)
- Non-JSON bodies require exact string match
- After scenario, unused stubs throw an exception

**Database Isolation:**
- `@BeforeScenario`: ORMPurger truncates all tables
- `@BeforeStep`: Entity manager cleared to prevent cached entity issues

| Context | Purpose | Key Hook |
|---------|---------|----------|
| RestApiContext | HTTP request/response testing | - |
| JwtContext | JWT token management | `@BeforeScenario` resets token |
| HttpClientContext | Stub external HTTP calls | `@AfterStep` verifies all stubs used |
| DoctrineORMContext | Database cleanup | `@BeforeScenario` purges DB |
| CliContext | Console command execution | - |

## Architecture

```
src/
├── RestApi/
│   ├── RestApiContext.php              # HTTP request/assertion steps
│   └── HttpAuthorizationTokenProvider.php  # Shared JWT token storage
├── HttpClient/
│   ├── HttpClientContext.php           # Stubbed HTTP interaction steps
│   ├── HttpInteraction.php             # Request/response parser
│   ├── MockClientCallback.php          # Mock HTTP client implementation
│   └── ResponseOverride.php            # (deprecated)
├── Jwt/
│   └── JwtContext.php                  # JWT token request/setup steps
├── Doctrine/
│   └── DoctrineORMContext.php          # Database cleanup hooks
└── Cli/
    └── CliContext.php                  # Console command execution steps
```

**Class Collaboration:**
```
HttpAuthorizationTokenProvider (shared state)
        │
        ├── RestApiContext ──── injects token into requests
        └── JwtContext ──────── stores token after login

MockClientCallback (HTTP stub registry)
        │
        └── HttpClientContext ── registers and verifies stubs
```

## Gherkin Steps Reference

### RestApiContext

#### Given I have the payload:

Stores the request body for the next HTTP request. The payload is preserved until consumed by a request step.

```gherkin
Given I have the payload:
  """
  {
    "name": "John Doe",
    "email": "john@example.com"
  }
  """
When I request "POST /api/users"
```

---

#### Given I have the headers:

Adds custom HTTP headers to subsequent requests. Headers are specified as a table with `Header` and `Value` columns.

```gherkin
Given I have the headers:
  | Header       | Value            |
  | X-Api-Key    | secret-key-123   |
  | Accept-Language | en-US         |
When I request "GET /api/protected-resource"
```

---

#### When I request "(GET|POST|DELETE) \<path\>"

Executes an HTTP request to the Symfony application. Automatically sets `Content-Type: application/json` and injects JWT token if available.

```gherkin
# Simple GET request
When I request "GET /api/users"

# POST request with previously set payload
Given I have the payload:
  """
  {"name": "New User"}
  """
When I request "POST /api/users"

# DELETE request
When I request "DELETE /api/users/123"
```

With inline body (PyStringNode):
```gherkin
When I request "POST /api/users"
  """
  {"name": "John", "email": "john@example.com"}
  """
```

---

#### When I send the payload "(GET|PUT|POST) \<path\>":

Combines payload definition and request execution in a single step. Supports GET, PUT, and POST methods.

```gherkin
When I send the payload "POST /api/users":
  """
  {
    "name": "Jane Doe",
    "email": "jane@example.com"
  }
  """

When I send the payload "PUT /api/users/1":
  """
  {
    "name": "Jane Smith"
  }
  """
```

---

#### Then the response status code should be \<code\>

Asserts the HTTP status code of the last response.

```gherkin
When I request "GET /api/users"
Then the response status code should be 200

When I request "POST /api/users"
  """
  {"name": "John"}
  """
Then the response status code should be 201

When I request "GET /api/nonexistent"
Then the response status code should be 404
```

---

#### Then the json response should be:

Asserts the JSON response body. Comparison is semantic (key order doesn't matter, whitespace is normalized).

```gherkin
When I request "GET /api/users/1"
Then the response status code should be 200
And the json response should be:
  """
  {
    "id": 1,
    "name": "John Doe",
    "email": "john@example.com"
  }
  """
```

---

#### Then the response should be:

Asserts the plain text response body (exact match after trimming).

```gherkin
When I request "GET /api/health"
Then the response status code should be 200
And the response should be:
  """
  OK
  """
```

---

### JwtContext

#### Given I request my token with "(GET|POST) \<path\>" using "\<username\>" as username and "\<password\>" as password

Performs authentication request and stores the JWT token. The token is automatically added to all subsequent requests as `Authorization: Bearer <token>`.

```gherkin
Given I request my token with "POST /api/login" using "admin@example.com" as username and "password123" as password
When I request "GET /api/protected-resource"
Then the response status code should be 200
```

Full scenario example:
```gherkin
Scenario: Access protected endpoint with valid token
  Given I request my token with "POST /api/login" using "user@test.com" as username and "secret" as password
  When I request "GET /api/profile"
  Then the response status code should be 200
  And the json response should be:
    """
    {"id": 1, "email": "user@test.com"}
    """
```

---

#### Given I have invalid JWT

Sets an intentionally invalid JWT token to test 401 Unauthorized responses.

```gherkin
Scenario: Reject invalid token
  Given I have invalid JWT
  When I request "GET /api/protected-resource"
  Then the response status code should be 401
```

---

### HttpClientContext

#### Given there is a stubbed http interaction:

Defines a mock HTTP interaction for external API calls. Uses a special format with `>>>` for request and `<<<` for response.

**Format:**
```
>>>
<METHOD> <URL>
<Header>: <Value>
<Request Body>
<<<
HTTP/<version> <status> <message>
<Header>: <Value>
<Response Body>
```

**Basic example:**
```gherkin
Given there is a stubbed http interaction:
  """
  >>>
  GET https://api.external.com/users/1
  <<<
  HTTP/2 200 OK
  Content-Type: application/json
  {"id": 1, "name": "External User"}
  """
When I request "GET /api/fetch-external-user/1"
Then the response status code should be 200
```

**POST with request body matching:**
```gherkin
Given there is a stubbed http interaction:
  """
  >>>
  POST https://payment.gateway.com/charge
  Content-Type: application/json
  {"amount": 100, "currency": "USD"}
  <<<
  HTTP/2 201 Created
  Content-Type: application/json
  {"transaction_id": "txn_123", "status": "success"}
  """
When I send the payload "POST /api/checkout":
  """
  {"cart_id": 1}
  """
Then the response status code should be 200
```

**Multiple interactions in one scenario:**
```gherkin
Scenario: Aggregate data from multiple external services
  Given there is a stubbed http interaction:
    """
    >>>
    GET https://service-a.com/data
    <<<
    HTTP/2 200 OK
    {"value": 10}
    """
  And there is a stubbed http interaction:
    """
    >>>
    GET https://service-b.com/data
    <<<
    HTTP/2 200 OK
    {"value": 20}
    """
  When I request "GET /api/aggregate"
  Then the response status code should be 200
  And the json response should be:
    """
    {"total": 30}
    """
```

**Error response simulation:**
```gherkin
Given there is a stubbed http interaction:
  """
  >>>
  GET https://api.external.com/unavailable
  <<<
  HTTP/2 503 Service Unavailable
  {"error": "Service temporarily unavailable"}
  """
When I request "GET /api/proxy-external"
Then the response status code should be 502
```

---

### DoctrineORMContext

No explicit Gherkin steps. Provides automatic hooks:

- **@BeforeScenario**: Purges all database tables using Doctrine ORMPurger
- **@BeforeStep**: Clears entity manager to prevent stale entity issues

```gherkin
# Database is automatically cleaned before this scenario runs
Scenario: Create and retrieve user
  When I send the payload "POST /api/users":
    """
    {"name": "Test User", "email": "test@example.com"}
    """
  Then the response status code should be 201
  When I request "GET /api/users"
  Then the json response should be:
    """
    [{"id": 1, "name": "Test User", "email": "test@example.com"}]
    """
```

---

### CliContext

#### Given I execute the "\<command\>" command

Executes a Symfony console command in the test environment.

```gherkin
Given I execute the "app:import-data" command
When I request "GET /api/imported-items"
Then the response status code should be 200
```

---

#### Given I execute the "\<command\>" command with "\<arguments\>" arguments

Executes a console command with arguments. Arguments are parsed as CSV (space-separated, quotes supported).

```gherkin
# Single argument
Given I execute the "app:process-user" command with "123" arguments

# Multiple arguments
Given I execute the "app:sync" command with "--force --limit=100" arguments

# Arguments with spaces (use quotes)
Given I execute the "app:notify" command with "'Hello World' user@example.com" arguments
```

Full scenario example:
```gherkin
Scenario: Process queued items via command
  Given I send the payload "POST /api/queue":
    """
    {"task": "send-email", "to": "user@test.com"}
    """
  And I execute the "messenger:consume" command with "async --limit=1" arguments
  When I request "GET /api/queue/status"
  Then the json response should be:
    """
    {"pending": 0, "processed": 1}
    """
```

---

## Complete Scenario Examples

### API CRUD Operations

```gherkin
Feature: User API

  Scenario: Full CRUD lifecycle
    # Create
    When I send the payload "POST /api/users":
      """
      {"name": "John", "email": "john@example.com"}
      """
    Then the response status code should be 201
    And the json response should be:
      """
      {"id": 1, "name": "John", "email": "john@example.com"}
      """

    # Read
    When I request "GET /api/users/1"
    Then the response status code should be 200

    # Update
    When I send the payload "PUT /api/users/1":
      """
      {"name": "John Smith"}
      """
    Then the response status code should be 200

    # Delete
    When I request "DELETE /api/users/1"
    Then the response status code should be 204
```

### Authenticated API with External Service

```gherkin
Feature: Payment processing

  Scenario: Process payment with external gateway
    Given I request my token with "POST /api/login" using "merchant@shop.com" as username and "secret" as password
    And there is a stubbed http interaction:
      """
      >>>
      POST https://payments.stripe.com/v1/charges
      Content-Type: application/json
      {"amount": 5000, "currency": "usd", "source": "tok_visa"}
      <<<
      HTTP/2 200 OK
      Content-Type: application/json
      {"id": "ch_123", "status": "succeeded", "amount": 5000}
      """
    When I send the payload "POST /api/orders/1/pay":
      """
      {"payment_token": "tok_visa"}
      """
    Then the response status code should be 200
    And the json response should be:
      """
      {"order_id": 1, "payment_status": "paid", "charge_id": "ch_123"}
      """
```

---

## Installation

```bash
composer require avodel/symfony-behat-api
```
