<?php

declare(strict_types=1);

namespace Avodel\SymfonyBehatApi\HttpClient;

class HttpInteraction
{
    public function __construct(
        private readonly string $url,
        private readonly string $method,
        private readonly int $statusCode,
        private readonly array $requestHeaders = [],
        private readonly string $requestBody = '',
        private readonly array $responseHeaders = [],
        private readonly string $responseBody = '',
    ) {
    }

    public function getUrl(): string
    {
        return $this->url;
    }

    public function getMethod(): string
    {
        return $this->method;
    }

    public function getStatusCode(): int
    {
        return $this->statusCode;
    }

    public function getRequestHeaders(): array
    {
        return $this->requestHeaders;
    }

    public function getRequestBody(): string
    {
        return $this->requestBody;
    }

    public function getResponseHeaders(): array
    {
        return $this->responseHeaders;
    }

    public function getResponseBody(): string
    {
        return $this->responseBody;
    }

    public static function createFromString(string $string): self
    {
        $input = trim($string);

        $parts = preg_split('/\s*>>>\s*|\s*<<<\s*/', $input);
        if (count($parts) !== 3) {
            $errorText = <<<EOT
Input string does not contain valid request and response data. Example: 
>>>
POST https://examples.http-client.intellij.net/post
Content-Type: application/json
{
    "id": 1,
    "value": "content"
}
<<<
HTTP/2 200 OK
Content-Type: application/json
{
    "id": 1,
    "value": "content"
}
EOT;
            throw new \InvalidArgumentException($errorText);
        }

        $requestString = trim($parts[1]);
        $requestString = str_replace(["\r\n", "\r"], "\n", $requestString);

        $requestLines = array_map('trim', explode("\n", $requestString));

        if (!preg_match('/^(POST|GET|PUT|DELETE|PATCH)\s+([^\s]+)$/', $requestLines[0], $requestLine)) {
            throw new \RuntimeException('Failed to parse request: invalid request line');
        }

        [, $method, $url] = $requestLine;

        array_shift($requestLines);

        $requestHeaders = [];
        $requestBody = '';
        $isHeader = true;

        foreach ($requestLines as $line) {
            if ($isHeader) {
                if (preg_match('/^([^:]+):\s*(.*)$/', $line)) {
                    $requestHeaders[] = $line;
                } else {
                    $isHeader = false;
                    $requestBody .= $line . "\n";
                }
                continue;
            }

            $requestBody .= $line . "\n";
        }

        json_decode($requestBody);
        if (
            json_last_error() === JSON_ERROR_NONE
            && !preg_grep('/^Content-Type:\s*application\/[\w.+-]*json\b/i', $requestHeaders)
        ) {
            $requestHeaders[] = 'Content-Type: application/json';
        }

        $responseString = trim($parts[2]);
        $responseString = str_replace(["\r\n", "\r"], "\n", $responseString);

        $responseLines = array_map('trim', explode("\n", $responseString));

        $statusLine = array_shift($responseLines);
        $statusCode = 200;

        if (preg_match('/^HTTP\/\d(?:\.\d)?\s+(\d{3})\s*/', $statusLine, $matches)) {
            $statusCode = (int) $matches[1];
        }

        $responseHeaders = [];
        $responseBody = '';
        $isHeader = true;

        foreach ($responseLines as $line) {
            if ($isHeader) {
                if (preg_match('/^([^:]+):\s*(.*)$/', $line, $matches)) {
                    $responseHeaders[] = $line;
                } else {
                    $isHeader = false;
                    $responseBody .= $line . "\n";
                }
                continue;
            }

            $responseBody .= $line . "\n";
        }

        json_decode($responseBody);
        if (json_last_error() === JSON_ERROR_NONE && !in_array('Content-Type: application/json', $responseHeaders, true)) {
            $responseHeaders[] = 'Content-Type: application/json';
        }


        return new self(
            $url,
            $method,
            $statusCode,
            $requestHeaders,
            trim(str_replace(["\n"], '', $requestBody)),
            $responseHeaders,
            trim(str_replace(["\n"], '', $responseBody)),
        );
    }
}
