<?php

namespace Avodel\SymfonyBehatApi\HttpClient;

use Avodel\SymfonyBehatApi\HttpClient\HttpRequest as BehatRequest;
use RuntimeException;
use Symfony\Component\HttpClient\Response\MockResponse;
use Symfony\Contracts\HttpClient\ResponseInterface;

final class MockClientCallback
{
    /**
     * @deprecated  use $interactions and $contentableInteractions
     *
     * @var array<string, array<string, ResponseOverride>>
     */
    private array $overrides = [];

    /**
     * @var array<string, array<string, HttpInteraction>>
     */
    private array $interactions = [];

    /**
     * @var array<string, array<string, HttpInteraction>>
     */
    private array $usedInteractions = [];

    /**
     * @var array<string, array<string, array<string, array<string, HttpInteraction>>>>
     */
    private array $contentableInteractions = [];

    /**
     * @var array<string, array<string, array<string, array<string, HttpInteraction>>>>
     */
    private array $usedContentableInteractions = [];

    /**
     * @var array<string, string[]>
     */
    private array $sentJsonRequests = [];

    public function setHttpInteraction(HttpInteraction $httpInteraction): void
    {
        if ($httpInteraction->getRequestBody()) {
            $this->contentableInteractions[$httpInteraction->getMethod()][$httpInteraction->getUrl()][$this->getContentTypeFromHeaders($httpInteraction->getRequestHeaders())][$httpInteraction->getRequestBody()] = $httpInteraction;

            return;
        }

        $this->interactions[$httpInteraction->getMethod()][$httpInteraction->getUrl()] = $httpInteraction;
    }

    /**
     * @deprecated use setHttpInteraction method
     */
    public function setOverride(string $requestMethod, string $url, string $responseContent, int $responseStatusCode): void
    {
        $this->overrides[$requestMethod][$url] = new ResponseOverride($responseContent, $responseStatusCode);
    }

    public function __invoke(string $method, string $url, array $options = []): ResponseInterface
    {
        if (isset($this->contentableInteractions[$method][$url]) && is_array($this->contentableInteractions[$method][$url])) {
            return $this->handleContentableInteraction($method, $url, $options);
        }

        if (isset($this->interactions[$method][$url])) {
            return $this->handleInteraction($method, $url);
        }

        if (isset($this->overrides[$method][$url])) {
            return $this->handleDeprecatedOverride($method, $url, $options);
        }

        throw new RuntimeException('No override for ' . $method . ' ' . $url);
    }

    private function handleDeprecatedOverride(string $method, string $url, array $options = []): ResponseInterface
    {
        $override = $this->overrides[$method][$url];

        if (in_array('Content-Type: application/json', $options['headers'], true)) {
            $this->sentJsonRequests[$url][] = $options['body'] ?? '';
        }


        return new MockResponse($override->getBody(), ['http_code' => $override->getStatusCode()]);
    }

    private function handleContentableInteraction(string $method, string $url, array $options = []): ResponseInterface
    {
        $currentRequestBody = $options['body'] ?? '';
        if (preg_grep('/^Content-Type:\s*application\/[\w.+-]*json\b/i', $options['headers'])) {
            $this->sentJsonRequests[$url][] = $currentRequestBody;
        }

        $currentRequestContentType = $this->getContentTypeFromHeaders($options['headers'] ?? []);

        foreach ($this->contentableInteractions[$method][$url] as $overrideContentType => $bodyContents) {
            foreach ($bodyContents as $requestBody => $interaction) {
                if (($overrideContentType === $currentRequestContentType) && $this->isRequestBodyMatch($overrideContentType, $requestBody, $currentRequestBody)) {
                    $this->usedContentableInteractions[$method][$url][$overrideContentType][$requestBody] = $interaction;
                    return new MockResponse($interaction->getResponseBody(), ['http_code' => $interaction->getStatusCode(), 'response_headers' => $interaction->getResponseHeaders()]);
                }
            }

        }

        throw new RuntimeException('No interaction for ' . $method . ' ' . $url . ' with content type ' . $currentRequestContentType . ' with body ' . $currentRequestBody);
    }

    private function handleInteraction(string $method, string $url): ResponseInterface
    {
        $interaction = $this->interactions[$method][$url];
        $this->usedInteractions[$method][$url] = $interaction;

        return new MockResponse($interaction->getResponseBody(), ['http_code' => $interaction->getStatusCode(), 'response_headers' => $interaction->getResponseHeaders()]);
    }

    private function isRequestBodyMatch(string $contentType, string $requestBody, string $currentRequestBody): bool
    {
        if (preg_match('/^application\/[\w.+-]*json\b/i', $contentType)) {
            $currentRequestBodyData = json_decode($currentRequestBody, true, 512, JSON_THROW_ON_ERROR);
            $requestBodyData = json_decode($requestBody, true, 512, JSON_THROW_ON_ERROR);
            return $requestBodyData == $currentRequestBodyData;
        }

        return $currentRequestBody === $requestBody;
    }

    private function getContentTypeFromHeaders(array $headers): ?string
    {
        $headersMap = [];
        foreach ($headers as $header) {
            if (preg_match('/^([^:]+):\s*(.*)$/', $header, $contentType)) {
                $headersMap[$contentType[1]] = $contentType[2];
            }
        }

        return $headersMap['Content-Type'] ?? null;
    }

    /**
     * @return string[]
     */
    public function getSentJsonRequests(string $url): array
    {
        return $this->sentJsonRequests[$url] ?? [];
    }

    /**
     * @return array<string, array<string, ResponseOverride>>
     */
    public function getInteractions(): array
    {
        return $this->interactions;
    }

    /**
     * @return array<string, array<string, ResponseOverride>>
     */
    public function getUsedInteractions(): array
    {
        return $this->usedInteractions;
    }

    /**
     * @return  array<string, array<string, array<string, array<string, ResponseOverride>>>>
     */
    public function getContentableInteractions(): array
    {
        return $this->contentableInteractions;
    }

    /**
     * @return  array<string, array<string, array<string, array<string, ResponseOverride>>>>
     */
    public function getUsedContentableInteractions(): array
    {
        return $this->usedContentableInteractions;
    }
}
