<?php

declare(strict_types=1);

namespace Avodel\SymfonyBehatApi\RestApi;

use Behat\Behat\Context\Context;
use Behat\Gherkin\Node\PyStringNode;
use Behat\Gherkin\Node\TableNode;
use Exception;
use JsonException;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\KernelInterface;
use Webmozart\Assert\Assert;

class RestApiContext implements Context
{
    private ?Response $response = null;

    private string $payload = '';

    private array $server = [];

    public function __construct(
        private readonly KernelInterface $kernel,
        private readonly HttpAuthorizationTokenProvider $httpAuthorizationTokenProvider,
    ) {
    }

    /**
     * @When /^I request "(GET|POST|DELETE) ([^"]*)"$/
     *
     * @throws Exception
     */
    public function iRequest(string $method, string $path, ?PyStringNode $response = null): void
    {
        $this->server['CONTENT_TYPE'] = 'application/json';

        $token = $this->httpAuthorizationTokenProvider->getToken();

        if ($token) {
            $this->server['HTTP_AUTHORIZATION'] = 'Bearer ' . $token;
        }

        $this->response = $this->kernel->handle(
            Request::create($path, $method, [], [], [], $this->server, $response?->getRaw() ?? $this->payload)
        );
    }


    /**
     * @Then /^the json response should be:$/
     *
     * @throws JsonException
     */
    public function theJsonResponseShouldBe(PyStringNode $response): void
    {
        $expectedJson = json_encode(
            json_decode($response->getRaw(), true, 512, JSON_THROW_ON_ERROR),
            JSON_THROW_ON_ERROR
        );
        Assert::eq($this->getResponse()->getContent(), $expectedJson);
    }

    /**
     * @Then /^the response should be:$/
     *
     * @throws JsonException
     */
    public function theResponseShouldBe(PyStringNode $response): void
    {
        $expectedJson = trim($response->getRaw());
        Assert::eq(trim($this->getResponse()->getContent()), $expectedJson);
    }

    /**
     * @Then /^the response status code should be (?P<code>\d+)$/
     */
    public function theResponseStatusCodeShouldBe(int $statusCode): void
    {
        Assert::eq($this->response->getStatusCode(), $statusCode);
    }

    private function getResponse(): Response
    {
        Assert::notNull($this->response);

        return $this->response;
    }

    /**
     * @Given /^I have the payload:$/
     */
    public function iHaveThePayload(PyStringNode $string): void
    {
        $this->payload = $string->getRaw();
    }

    /**
     * @Given /^I have the headers:$/
     */
    public function iHaveHeaders(TableNode $table): void
    {
        foreach ($table as $row) {
            $this->server['HTTP_' . $row['Header']] = $row['Value'];
        }
    }

    /**
     * @When /^I send the payload "(GET|PUT|POST) ([^"]*)":$/
     */
    public function iSendThePayload(PyStringNode $string, string $method, string $path): void
    {
        $this->payload = $string->getRaw();
        $this->iRequest($method, $path);
    }
}
