<?php

declare(strict_types=1);

namespace Avodel\TelegramBotApi\DependencyInjection;

use Avodel\TelegramBotApi\Contract\InvoiceRepositoryInterface;
use Avodel\TelegramBotApi\Contract\ThreadRepositoryInterface;
use Avodel\TelegramBotApi\Contract\UserRepositoryInterface;
use Avodel\TelegramBotApi\IssueInvoiceFactory;
use Avodel\TelegramBotApi\PostPaymentProcessorsRegistry;
use Avodel\TelegramBotApi\PreCheckoutValidatorsRegistry;
use Symfony\Component\Config\Definition\ConfigurationInterface;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Extension\Extension;
use Symfony\Component\DependencyInjection\Loader\YamlFileLoader;
use Symfony\Component\Config\FileLocator;
use Symfony\Component\DependencyInjection\Reference;
use Avodel\TelegramBotApi\Contract\ExceptionHandlerInterface;

class TelegramBotApiExtension extends Extension
{
    public function load(array $configs, ContainerBuilder $container): void
    {
        $loader = new YamlFileLoader($container, new FileLocator(__DIR__ . '/../Resources/Config/'));
        $loader->load('services.yaml');

        /** @var ConfigurationInterface $configuration */
        $configuration = $this->getConfiguration([], $container);

        /** @var array<string,string|array> $config */
        $config = $this->processConfiguration($configuration, $configs);

        $container->setParameter('avodel.telegram_bot_api.token', $config['token']);
        $container->setParameter('avodel.telegram_bot_api.payments.provider_token', $config['payments']['provider_token'] ?? null);
        $container->setParameter('avodel.telegram_bot_api.payments.enable_basic_pre_checkout_validation', $config['payments']['enable_basic_pre_checkout_validation'] ?? true);
        $container->setParameter('avodel.telegram_bot_api.base_url', $config['base_url']);
        $container->setParameter('avodel.telegram_bot_api.user_class', $config['user_class']);
        $container->setParameter('avodel.telegram_bot_api.thread_class', $config['thread_class']);

        if (isset($config['payments'])) {
            $loader->load('payment_services.yaml');

            $postPaymentProcessorsRegistryDef = $container->findDefinition(PostPaymentProcessorsRegistry::class);
            $preCheckoutValidatorsRegistryDef = $container->findDefinition(PreCheckoutValidatorsRegistry::class);

            $paymentProcessorReferences = [];
            $validatorReferences = [];
            foreach ($config['payments']['products'] as $product => $settings) {
                $issueInvoiceFactoryServiceId = 'avodel.telegram_bot_' . $product . '_issue_invoice_factory';

                $container
                    ->register($issueInvoiceFactoryServiceId, IssueInvoiceFactory::class)
                    ->setArgument('$product', $product)
                    ->setAutowired(true);

                if (isset($settings['post_payment_processor'])) {
                    $paymentProcessorReferences[$product] = new Reference($settings['post_payment_processor']);
                }

                if (isset($settings['pre_checkout_validator'])) {
                    $validatorReferences[$product] = new Reference($settings['pre_checkout_validator']);
                }
            }

            $postPaymentProcessorsRegistryDef->setArgument('$processors', $paymentProcessorReferences);
            $preCheckoutValidatorsRegistryDef->setArgument('$validators', $validatorReferences);
        }

        if ($config['behat']) {
            $loader->load('behat_services.yaml');
        }

        $exceptionHandlerServiceId = $config['exception_handler'];
        $container->setAlias(ExceptionHandlerInterface::class, $exceptionHandlerServiceId);
        $container->setAlias(UserRepositoryInterface::class, $config['user_repository']);
        $container->setAlias(ThreadRepositoryInterface::class, $config['thread_repository']);

        if ($config['payments']) {
            $container->setAlias(InvoiceRepositoryInterface::class, $config['payments']['invoice_repository']);
        }

        if ($config['async']) {
            $loader->load('async_services.yaml');
        }

        $container->setAlias('avodel.telegram_bot_api.translator', $config['translator']);
    }

    public function getAlias(): string
    {
        return 'avodel_telegram_bot_api';
    }
}
