<?php

namespace Avodel\TelegramBotApi\Repository;

use Avodel\TelegramBotApi\Contract\ThreadInterface;
use Avodel\TelegramBotApi\Contract\ThreadRepositoryInterface;
use Avodel\TelegramBotApi\Exception\ThreadNotFoundException;
use Symfony\Component\Serializer\SerializerInterface;
use Symfony\Contracts\Cache\CacheInterface;

final class KeyValueThreadRepository implements ThreadRepositoryInterface
{
    public function __construct(
        private readonly CacheInterface $cache,
        private readonly SerializerInterface $serializer,
        private readonly string $token,
        private readonly string $threadClass,
    )
    {
    }

    private function getMessageIdKey(int $messageId): string
    {
        [$botId, ] = explode(':', $this->token);

        return $botId . '_' . $messageId;
    }

    public function save(ThreadInterface $thread): void
    {
        $serializedThread = $this->serializer->serialize($thread, 'json');

        $this->cache->delete($thread->getId());
        $this->cache->get($thread->getId(), fn() => $serializedThread);

        if ($thread->getMessageId()) {
            $key = $this->getMessageIdKey($thread->getMessageId());
            $this->cache->delete($key);
            $this->cache->get($key, fn() => $thread->getId());
        }
    }

    public function getByMessageId(int $messageId): ThreadInterface
    {
        $key = $this->getMessageIdKey($messageId);
        /** @var string|null $threadId */
        $threadId = $this->cache->get($key, fn() => null);

        if (!$threadId) {
            throw new ThreadNotFoundException('Thread not found. Message ID: ' . $messageId);
        }

        /** @var string|null $data */
        $data = $this->cache->get($threadId, fn() => null);

        if (!$data) {
            throw new ThreadNotFoundException('Thread not found. Message ID: ' . $messageId);
        }

        return $this->serializer->deserialize($data, $this->threadClass, 'json');
    }

    public function findOneById(string $id): ?ThreadInterface
    {
        /** @var string|null $data */
        $data = $this->cache->get($id, fn() => null);

        if (!$data) {
            throw new ThreadNotFoundException('Thread not found. ID: ' . $id);
        }

        return $this->serializer->deserialize($data, $this->threadClass, 'json');
    }

}
