<?php

declare(strict_types=1);

namespace Avodel\TelegramBotApi;

use Avodel\TelegramBotApi\Contract\IssueInvoiceInterface;
use Avodel\TelegramBotApi\Contract\MessageInterface;
use Avodel\TelegramBotApi\Contract\PhotoInterface;
use Avodel\TelegramBotApi\Contract\SentMessageInterface;
use Avodel\TelegramBotApi\Contract\TelegramApiInterface;
use Avodel\TelegramBotApi\Exception\AnswerPreCheckoutQueryException;
use Avodel\TelegramBotApi\Exception\DeleteMessageException;
use Avodel\TelegramBotApi\Exception\GetChatMemberException;
use Avodel\TelegramBotApi\Exception\GetUpdatesException;
use Avodel\TelegramBotApi\Exception\IdenticalContentException;
use Avodel\TelegramBotApi\Exception\SendInvoiceMessageException;
use Avodel\TelegramBotApi\Exception\SendPhotoException;
use Avodel\TelegramBotApi\Exception\SendTelegramMessageException;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Exception\RequestException;
use JsonException;
use Psr\Log\LoggerInterface;

final class TelegramApi implements TelegramApiInterface
{
    private const TIMEOUT_SEC = 15;

    private const MAX_RETRIES_NO_CONNECT_ERROR = 3;

    public function __construct(
        private readonly string $token,
        private readonly LoggerInterface $telegramApiLogger,
        private readonly CallbackQueryConvertor $callbackQueryConvertor,
        private readonly string $baseUrl,
        private readonly ?string $paymentProviderToken = null,
    ) {
    }

    /**
     * @throws JsonException
     */
    public function sendMessage(int $chatId, MessageInterface $message): SentMessageInterface
    {
        $data = [
            'chat_id' => $chatId,
            'text' => $message->getText(),
            'parse_mode' => $message->getParseMode()?->value,
            'disable_web_page_preview' => true,
        ];

        if ($message->getInlineKeyboardLines()) {
            $keyboard = json_encode(['inline_keyboard' => $this->normalizeKeyboard($message)], JSON_THROW_ON_ERROR);
            $data['reply_markup'] = $keyboard;
        }

        $response = $this->send('sendMessage', $data);

        if (!$response) {
            throw new SendTelegramMessageException('Text message was not sent.');
        }

        $this->telegramApiLogger->info('Telegram message was sent.', ['chatId' => $chatId]);

        return new SentMessage($response['result']['message_id']);
    }

    /**
     * @throws JsonException
     */
    public function editMessage(int $chatId, int $messageId, MessageInterface $message): void
    {
        $data = [
            'chat_id' => $chatId,
            'message_id' => $messageId,
            'text' => $message->getText(),
            'parse_mode' => $message->getParseMode()?->value,
            'disable_web_page_preview' => true,
        ];

        if ($message->getInlineKeyboardLines()) {
            $keyboard = json_encode(['inline_keyboard' => $this->normalizeKeyboard($message)], JSON_THROW_ON_ERROR);
            $data['reply_markup'] = $keyboard;
        }

        try {
            if (!$this->send('editMessageText', $data)) {
                throw new SendTelegramMessageException('Text message was not sent.');
            }
        } catch (IdenticalContentException $e) {
            $this->telegramApiLogger->warning('Unable to edit message because the content is identical.', [
                'chatId' => $chatId,
                'messageId' => $messageId,
                'exception' => $e,
            ]);

            return;
        }

        $this->telegramApiLogger->info('Telegram message was edited.', [
            'chatId' => $chatId,
        ]);
    }

    /**
     * @throws JsonException
     */
    public function sendInvoice(int $chatId, IssueInvoiceInterface $invoice): void
    {
        if (!$this->paymentProviderToken) {
            throw new SendInvoiceMessageException('Payment provider token is undefined. Set it to use invoice API');
        }

        $totalAmount = $invoice->getPrice()->getAmount();

        $data = [
            'chat_id' => $chatId,
            'title' => $invoice->getTitle(),
            'description' => $invoice->getDescription(),
            'payload' => $invoice->getPayload(),
            'provider_token' => $this->paymentProviderToken,
            'currency' => $invoice->getCurrency(),
            'start_parameter' => $invoice->getPayload(),
            'prices' => json_encode($this->normalizeInvoicePrices($invoice), JSON_THROW_ON_ERROR)
        ];

        if ($totalAmount > 3000) {
            $data['suggested_tip_amounts'] = json_encode([1000, 1500, 2000, 3000], JSON_THROW_ON_ERROR);
            $data['max_tip_amount'] = $totalAmount;
        }

        if (!$this->send('sendInvoice', $data)) {
            throw new SendInvoiceMessageException('Invoice was not sent.');
        }

        $this->telegramApiLogger->info('Invoice was sent.', [
            'chatId' => $chatId,
            'payload' => $invoice->getPayload(),
        ]);
    }

    /**
     * @throws JsonException
     */
    public function answerPreCheckoutQuery(string $preCheckoutQueryId, bool $success): void
    {
        $this->telegramApiLogger->info('Sending answerPreCheckoutQuery.', [
            'preCheckoutQueryId' => $preCheckoutQueryId,
            'success' => $success,
        ]);

        $data = [
            'pre_checkout_query_id' => $preCheckoutQueryId,
            'ok' => $success,
        ];

        if (!$this->send('answerPreCheckoutQuery', $data)) {
            throw new AnswerPreCheckoutQueryException('AnswerPreCheckoutQuery was not sent.');
        }

        $this->telegramApiLogger->info('The answerPreCheckoutQuery was sent.', [
            'preCheckoutQueryId' => $preCheckoutQueryId,
            'success' => $success,
        ]);
    }

    public function getUpdates(int $offset = 0): array
    {
        $updates = $this->send('getUpdates', [
            'offset' => $offset,
        ]);

        if (!$updates) {
            throw new GetUpdatesException('getUpdates failed.');
        }

        $this->telegramApiLogger->info('Got telegram updates.');

        return $updates['result'];
    }

    /**
     * @throws JsonException
     * @throws IdenticalContentException
     */
    private function send(string $action, array $data, int $retryNumber = 0): array
    {
        $this->telegramApiLogger->debug('Sending request to Telegram.', [
            'action' => $action,
        ]);

        $client = new Client();
        $response = null;

        try {
            if ($action === 'sendPhoto') {
                $response = $client->post(sprintf($this->baseUrl . '/bot%s/%s', $this->token, $action), [
                    'multipart' => $data,
                    'connect_timeout' => self::TIMEOUT_SEC,
                    'read_timeout' => self::TIMEOUT_SEC,
                ]);
            } else {
                $response = $client->post(sprintf($this->baseUrl . '/bot%s/%s', $this->token, $action), [
                    'form_params' => $data,
                    'connect_timeout' => self::TIMEOUT_SEC,
                    'read_timeout' => self::TIMEOUT_SEC,
                ]);
            }

            $response = $response->getBody()->getContents();
        } catch (RequestException $e) {
            if ($e->hasResponse()) {
                $body = $e->getResponse()->getBody()->getContents();
                $this->telegramApiLogger->error('Failed response from Telegram.', [
                    'body' => $body,
                    'action' => $action,
                ]);

                if (str_contains($body, 'specified new message content and reply markup are exactly the same as a current content')) {
                    throw new IdenticalContentException('specified new message content and reply markup are exactly the same as a current content');
                }

                return [];
            }

            $this->telegramApiLogger->error('The request to telegram was not succeeded.', [
                'action' => $action,
            ]);

            return [];
        } catch (GuzzleException $e) {
            $this->telegramApiLogger->error('Failed to send message to Telegram.', [
                'action' => $action,
                'exceptionClass' => get_class($e),
                'retryNumber' => $retryNumber,
            ]);

            if ($retryNumber > self::MAX_RETRIES_NO_CONNECT_ERROR) {
                $this->telegramApiLogger->error('Telegram API ran out of retries.', [
                    'action' => $action,
                    'retryNumber' => $retryNumber,
                ]);

                return [];
            }

            $this->telegramApiLogger->error('Retrying sending request to Telegram.', [
                'action' => $action,
                'retryNumber' => $retryNumber,
            ]);

            return $this->send($action, $data, ++$retryNumber);
        } finally {
            $this->telegramApiLogger->debug('Finished sending request to Telegram.', [
                'action' => $action,
            ]);
        }

        if (!$response) {
            $this->telegramApiLogger->error('Blank response from Telegram.', [
                'action' => $action,
            ]);

            return [];
        }

        try {
            $jsonResponse = json_decode($response, true, 512, JSON_THROW_ON_ERROR);
        } catch (JsonException $e) {
            $this->telegramApiLogger->error('Unable to decode response from Telegram.', [
                'action' => $action,
                'json' => $response,
                'exception' => $e,
            ]);

            return [];
        }

        if (!$jsonResponse['ok']) {
            $this->telegramApiLogger->error('Failed response from Telegram.', [
                'action' => $action,
                'response' => json_encode($jsonResponse, JSON_THROW_ON_ERROR),
            ]);

            return [];
        }

        return $jsonResponse;
    }

    private function normalizeInvoicePrices(IssueInvoiceInterface $invoice): array
    {
        return [[
            'label' => $invoice->getPrice()->getLabel(),
            'amount' => $invoice->getPrice()->getAmount(),
        ]];
    }

    /**
     * @return array<array<array>>
     */
    private function normalizeKeyboard(MessageInterface $message): array
    {
        $inlineKeyboard = [];

        foreach ($message->getInlineKeyboardLines() as $lineIndex => $inlineKeyboardLines) {
            foreach ($inlineKeyboardLines as $columnIndex => $button) {
                $inlineKeyboard[$lineIndex][$columnIndex] = [
                    'text' => $button->getText(),
                    'callback_data' => $this->callbackQueryConvertor->convertButtonToCallbackData($button),
                ];
            }
        }

        return $inlineKeyboard;
    }

    public function sendPhoto(int $chatId, PhotoInterface $photo): void
    {
        $data = [
            [
                'name' => 'chat_id',
                'contents' => $chatId
            ],
            [
                'name' => 'photo',
                'contents' => fopen($photo->getPhotoPath(), 'r'),
                'filename' => basename($photo->getPhotoPath()),
            ],
            [
                'name' => 'caption',
                'contents' => $photo->getCaption()
            ]
        ];

        if (!$this->send('sendPhoto', $data)) {
            throw new SendPhotoException('Photo was not sent.');
        }

        $this->telegramApiLogger->info('Photo sent.', [
            'photoPath' => $photo->getPhotoPath(),
            'caption' => $photo->getCaption(),
        ]);
    }

    public function getChatMember(int $chatId, int $userId): array
    {
        $data = [
            'chat_id' => $chatId,
            'user_id' => $userId,
        ];

        $result = $this->send('getChatMember', $data);

        if (!$result) {
            throw new GetChatMemberException('Getting chat member message was sent.');
        }

        $this->telegramApiLogger->debug('Getting chat member message was sent.', [
            'chatId' => $chatId,
            'userId' => $userId
        ]);

        return $result;
    }

    public function deleteMessage(int $chatId, int $messageId): void
    {
        $data = [
            'chat_id' => $chatId,
            'message_id' => $messageId,
        ];

        $result = $this->send('deleteMessage', $data);

        if (!$result) {
            throw new DeleteMessageException('Unable to delete message.');
        }

        $this->telegramApiLogger->debug('Telegram message was deleted.', [
            'chatId' => $chatId,
            'messageId' => $messageId
        ]);
    }
}
