<?php

declare(strict_types=1);

namespace Avodel\TelegramBotApi;

use Avodel\TelegramBotApi\Contract\CallbackQueryHandlerInterface;
use Avodel\TelegramBotApi\Contract\TextMessageHandlerInterface;
use Avodel\TelegramBotApi\Exception\TextMessageHandlerNotFoundException;
use Symfony\Component\DependencyInjection\Attribute\TaggedIterator;
use Webmozart\Assert\Assert;

final class TextMessageHandlerRegistry
{
    /**
     * @var array<string, TextMessageHandlerInterface>
     */
    private array $handlers = [];

    /**
     * @param iterable|TextMessageHandlerInterface[] $handlers
     */
    public function __construct(
        iterable $handlers
    ){
        foreach ($handlers as $handler) {
            $this->register($handler);
        }
    }

    public function register(TextMessageHandlerInterface $handler): void
    {
        foreach ($handler->getSubscribedCallbacks() as $subscribedCallbackDatum) {
            Assert::keyNotExists($this->handlers, $subscribedCallbackDatum);
            $this->handlers[$subscribedCallbackDatum] = $handler;
        }
    }

    /**
     * @throws TextMessageHandlerNotFoundException
     */
    public function getHandler(string $callbackData): TextMessageHandlerInterface
    {
        if (!isset($this->handlers[$callbackData])) {
            throw TextMessageHandlerNotFoundException::fromCallbackData($callbackData);
        }

        return $this->handlers[$callbackData];
    }
}
