<?php

namespace Avodel\TelegramBotApi;

use Avodel\TelegramBotApi\Contract\InvoiceInterface;
use Avodel\TelegramBotApi\Contract\InvoiceRepositoryInterface;
use Avodel\TelegramBotApi\Contract\IssueInvoiceInterface;
use Avodel\TelegramBotApi\Contract\MessageInterface;
use Avodel\TelegramBotApi\Contract\ParseMode;
use Avodel\TelegramBotApi\Contract\ReadonlyUserInterface;
use Avodel\TelegramBotApi\Contract\SentMessageInterface;
use Avodel\TelegramBotApi\Contract\TelegramApiInterface;
use Avodel\TelegramBotApi\Contract\ThreadInteractorInterface;
use Avodel\TelegramBotApi\Contract\ThreadInterface;
use Avodel\TelegramBotApi\Contract\ThreadStateInterface;
use Avodel\TelegramBotApi\Contract\UserInterface;
use Symfony\Contracts\Translation\TranslatorInterface;

class ThreadInteractor implements ThreadInteractorInterface
{
    public function __construct(
        private readonly TelegramApiInterface $telegramApi,
        private readonly TranslatorInterface $translator,
        private readonly ThreadInterface $thread,
        private readonly UserInterface $user,
        private readonly ?InvoiceRepositoryInterface $invoiceRepository = null,
        private readonly ?InvoiceFactory $invoiceFactory = null,
    )
    {
    }

    public function getUser(): ReadonlyUserInterface
    {
        return $this->user;
    }

    public function writeState(ThreadStateInterface $state): void
    {
        $serializedState = $state->serialize();

        $this->thread->setStateItem($state->getKey(), $serializedState);
    }

    /**
     * @inheritDoc
     */
    public function unsetState(string $className): void
    {
        $this->thread->unsetState($className::getKey());
    }

    /**
     * @inheritdoc
     */
    public function readState(string $className): ThreadStateInterface
    {
        return $className::deserialize($this->thread->getState()[$className::getKey()] ?? []);
    }

    public function reply(MessageInterface $message): SentMessageInterface
    {
        $translatedMessage = $this->getTranslatedMessage($message);

        if (!$this->thread->getMessageId()) {
            $sentMessage = $this->telegramApi->sendMessage($this->user->getId(), $translatedMessage);
        } else {
            $this->telegramApi->editMessage($this->user->getId(), $this->thread->getMessageId(), $translatedMessage);
        }

        $this->registerCallbacks($message);

        $result = $sentMessage ?? new SentMessage($this->thread->getMessageId());

        $this->thread->setMessageId($result->getMessageId());

        return $result;
    }

    private function registerCallbacks(MessageInterface $message): void
    {
        $this->user->setTextCallback($message->getCallback());
        $expectedCallbacks = [];
        foreach ($message->getInlineKeyboardLines() as $keyboardLines) {
            foreach ($keyboardLines as $button) {
                $expectedCallbacks[] = $button->getCallbackData();
            }
        }

        $this->thread->setQueryCallBacks(array_values(array_unique($expectedCallbacks)));
    }

    private function getTranslatedMessage(MessageInterface $message): MessageInterface
    {
        $languageCode = $this->user->getLanguageCode();

        $translatedParameters = array_map(
            static function (int|string|null $parameter) use ($message) {
                $param = (string)$parameter;

                return $message->getParseMode() === ParseMode::MARKDOWN_V2
                    ? MarkdownHelper::escape($param)
                    : $param;
            },

            $message->getTranslationParameters()
        );

        $translatedMessage = $this->translator->trans(
            $message->getText(),
            $translatedParameters,
            null,
            $languageCode
        );

        $inlineKeyboard = $message->getInlineKeyboardLines();

        foreach ($inlineKeyboard as &$keyboardLines) {
            foreach ($keyboardLines as &$button) {
                $buttonText = $this->translator->trans(
                    $button->getText(),
                    $button->getTranslationParameters(),
                    null,
                    $languageCode
                );

                $button = $button->withText($buttonText);
            }
        }

        return new Message(
            $translatedMessage,
            $inlineKeyboard,
            $translatedParameters,
            $message->getCallback(),
            $message->getParseMode(),
        );
    }

    public function issueInvoice(IssueInvoiceInterface $issueInvoice): InvoiceInterface
    {
        if (!$this->invoiceFactory instanceof InvoiceFactory) {
            throw new \RuntimeException('Invoice factory is not set. Probably payment was not configured');
        }

        if (!$this->invoiceRepository instanceof InvoiceRepositoryInterface) {
            throw new \RuntimeException('Invoice repository was not instantiated. Instantiate it first if you are using payments');
        }

        $languageCode = $this->user->getLanguageCode();
        $params = $issueInvoice->getTranslationParameters();
        $translatedTitle = $this->translator->trans($issueInvoice->getTitle(), $params, null, $languageCode);
        $translatedDescription = $this->translator->trans($issueInvoice->getDescription(), $params, null, $languageCode);

        $price = $issueInvoice->getPrice();
        $translatedLabel = $this->translator->trans($price->getLabel(), $params, null, $languageCode);
        $translatedPrice = new LabeledPrice($translatedLabel, $price->getAmount());

        $this->telegramApi->sendInvoice(
            $this->user->getId(),
            new IssueInvoice(
                $translatedTitle,
                $translatedDescription,
                $issueInvoice->getPayload(),
                $issueInvoice->getCurrency(),
                $translatedPrice,
                $issueInvoice->getProduct()
            )
        );

        $invoice = $this->invoiceFactory->create(
            $issueInvoice->getPayload(),
            $issueInvoice->getProduct(),
            $this->user->getId(),
            $translatedPrice->getAmount(),
            $issueInvoice->getCurrency()
        );

        $this->invoiceRepository->save($invoice);

        return $invoice;
    }
}
