<?php

declare(strict_types=1);

namespace Avodel\TelegramBotApi;

use Avodel\TelegramBotApi\Exception\NoSupportedUpdateHandlerException;
use Avodel\TelegramBotApi\UpdateHandler\UpdateHandlerRegistry;
use Exception;
use Psr\Log\LoggerInterface;

final class UpdateHandler
{
    public function __construct(
        private readonly LoggerInterface $telegramApiLogger,
        private readonly UpdateHandlerRegistry $updateHandlerRegistry
    ) {
    }

    public function handleUpdate(array $update): void
    {
        $updateName = array_key_last($update);
        $chatId = $update[$updateName]['from']['id'] ?? null;
        $this->telegramApiLogger->debug('Received Telegram update.', [
            'updateName' => $updateName,
            'allKeys' => array_keys($update),
            'chatId' => $chatId,
        ]);

        try {
            $handler = $this->updateHandlerRegistry->getSupportedHandler($update);
            $this->telegramApiLogger->debug('Found supported handler.', [
                'handler' => get_class($handler),
                'updateName' => $updateName,
                'chatId' => $chatId,
            ]);
        } catch (NoSupportedUpdateHandlerException) {
            $this->telegramApiLogger->warning('Telegram update is not supported.', [
                'updateName' => $updateName,
                'chatId' => $chatId,
            ]);

            return;
        }

        try {
            $handler->handle($update);
        } catch (Exception $exception) {
            $this->telegramApiLogger->error('Telegram update handler failed with exception.', [
                'exception' => $exception,
                'handler' => get_class($handler),
                'updateName' => $updateName,
                'chatId' => $chatId,
            ]);
        }
    }
}
