<?php

declare(strict_types=1);

namespace Avodel\TelegramBotApi\UpdateHandler;

use Avodel\TelegramBotApi\CommandRegistry;
use Avodel\TelegramBotApi\Contract\ExceptionHandlerInterface;
use Avodel\TelegramBotApi\Contract\TelegramApiInterface;
use Avodel\TelegramBotApi\Contract\ThreadInteractorInterface;
use Avodel\TelegramBotApi\Contract\ThreadRepositoryInterface;
use Avodel\TelegramBotApi\Contract\UpdateHandlerInterface;
use Avodel\TelegramBotApi\Contract\UserInterface;
use Avodel\TelegramBotApi\Contract\UserNotFoundExceptionInterface;
use Avodel\TelegramBotApi\Contract\UserRepositoryInterface;
use Avodel\TelegramBotApi\Exception\CommandNotFoundException;
use Avodel\TelegramBotApi\Exception\DeleteMessageException;
use Avodel\TelegramBotApi\Exception\TextMessageHandlerNotFoundException;
use Avodel\TelegramBotApi\ReceivedTextMessage;
use Avodel\TelegramBotApi\TextMessageHandlerRegistry;
use Avodel\TelegramBotApi\ThreadFactory;
use Avodel\TelegramBotApi\ThreadInterfactorFactory;
use Avodel\TelegramBotApi\UserFactory;
use Psr\Log\LoggerInterface;
use Throwable;

final class TextMessageUpdateHandler implements UpdateHandlerInterface
{
    public function __construct(
        private readonly UserFactory $userFactory,
        private readonly LoggerInterface $logger,
        private readonly ExceptionHandlerInterface $exceptionHandler,
        private readonly UserRepositoryInterface $userRepository,
        private readonly CommandRegistry $commandRegistry,
        private readonly TextMessageHandlerRegistry $textMessageHandlerRegistry,
        private readonly ThreadInterfactorFactory $threadInterfactorFactory,
        private readonly ThreadRepositoryInterface $threadRepository,
        private readonly ThreadFactory $threadFactory,
        private readonly TelegramApiInterface $telegramApi
    )
    {
    }

    public function handle(array $update): void
    {
        $messageData = $update['message'];
        $text = trim($messageData['text']);
        $from = $messageData['from'];
        $messageId = $messageData['message_id'];
        $receivedMessage = new ReceivedTextMessage($messageId, $text);

        try {
            $user = $this->userRepository->get($from['id']);
        } catch (UserNotFoundExceptionInterface $e) {
            $this->logger->info('User not found. New one will be created.', ['user_id' => $from['id'], 'exception' => $e]);
            $user = $this->userFactory->create($from);
        }

        $thread = $this->threadFactory->createEmptyWithChatId($user->getId());

        try {
            if (str_starts_with($receivedMessage->getText(), '/')) {
                $threadInteractor = $this->threadInterfactorFactory->create($user, $thread);
                $this->handleCommand($receivedMessage, $threadInteractor);
            } else {
                $textCallBack = $user->getTextCallback();
                if (!$textCallBack) {
                    $this->logger->info('No text callback found', ['chat_id' => $user->getId()]);
                    return;
                }

                if (!$user->getLastInteractedThreadId()) {
                    $this->logger->warning('No last interacted thread id', ['chat_id' => $user->getId()]);
                    return;
                }

                $thread = $this->threadRepository->findOneById($user->getLastInteractedThreadId());

                if (!$thread) {
                    $this->logger->error('Unable to find last interacted thread', ['chat_id' => $user->getId()]);
                    return;
                }

                $threadInteractor = $this->threadInterfactorFactory->create($user, $thread);
                $messageToDelete = $thread->getMessageId();
                $thread->setMessageId(null);
                $this->handleTextMessage($receivedMessage, $threadInteractor, $user);
                try {
                    $this->telegramApi->deleteMessage($user->getId(), $messageToDelete);
                } catch (DeleteMessageException $e) {
                    $this->logger->notice('Unable to delete message, probably already deleted', ['message_id' => $messageToDelete, 'chat_id' => $user->getId(), 'exception' => $e]);
                }
            }

        } catch (CommandNotFoundException $e) {
            $this->logger->warning($e->getMessage(), ['exception' => $e]);

            return;
        } catch (Throwable $throwable) {
            $this->exceptionHandler->handle($update, $user->getId(), $throwable);
        }

        $this->threadRepository->save($thread);
        $user->setLastInteractedThreadId($thread->getId());
        $this->userRepository->save($user);
    }

    private function handleTextMessage(ReceivedTextMessage $message, ThreadInteractorInterface $thread, UserInterface $user): void
    {
        $textCallback = $user->getTextCallback();

        if (!$textCallback) {
            return;
        }

        try {
            $answerHandler = $this->textMessageHandlerRegistry->getHandler($textCallback);
        } catch (TextMessageHandlerNotFoundException) {
            $this->logger->error('Unable to find a text handler for callback.', [
                'callback' => $textCallback,
            ]);

            return;
        }

        $answerHandler->onTextMessage($message, $thread);
    }

    /**
     * @throws CommandNotFoundException
     */
    private function handleCommand(ReceivedTextMessage $message, ThreadInteractorInterface $thread): void
    {
        $commandName = substr($message->getText(), 1);
        $command = $this->commandRegistry->getCommand($commandName);
        $command->start($thread);
    }

    public function supports(array $update): bool
    {
        $messageData = $update['message'] ?? null;

        if (!$messageData) {
            return false;
        }

        if (!isset($messageData['text'])) {
            return false;
        }

        $message = trim($messageData['text']);

        if (!$message) {
            return false;
        }

        if (($messageData['chat']['type'] ?? null) !== 'private') {
            return false;
        }

        return true;
    }
}
