# Telegram Bot API Bundle

This document provides comprehensive guidance for integrating and using the `avodel/telegram-bot-api-bundle` in Symfony applications. It covers bundle integration, handler implementation patterns, state management, and testing strategies.

## Bundle Overview

The `avodel/telegram-bot-api-bundle` provides a robust framework for building Telegram bots with Symfony. It implements a registry-based handler architecture that routes Telegram updates (messages, callbacks, commands) to specialized handlers.

### Key Features

- **Command handling** - Process `/command` messages
- **Text message handling** - Process free-form text input
- **Callback query handling** - Handle inline keyboard button interactions
- **Thread-based state management** - Maintain conversation state across multiple messages
- **Payments integration** - Support for Telegram payments
- **Behat testing support** - BDD testing with stubs for Telegram API

---

## Installation and Configuration

### Step 1: Install the Bundle

```bash
composer require avodel/telegram-bot-api-bundle
```

### Step 2: Configure the Bundle

Create `config/packages/avodel_telegram_bot_api.yaml`:

```yaml
avodel_telegram_bot_api:
    token: '%env(TELEGRAM_BOT_API_TOKEN)%'
    user_repository: Avodel\TelegramBotApi\Repository\KeyValueUserRepository
    thread_repository: Avodel\TelegramBotApi\Repository\KeyValueThreadRepository
    user_class: App\Entity\User
    thread_class: App\Entity\Thread
    base_url: '%env(TELEGRAM_BASE_URL)%'
    translator: 'App\TranslationService\Translator'
    payments:
        provider_token: '%env(TELEGRAM_BOT_API_PAYMENT_PROVIDER_TOKEN)%'
        invoice_repository: App\Payments\InMemoryInvoiceRepository
        enable_basic_pre_checkout_validation: false
        products:
            '%env(TELEGRAM_PAYMENT_PRODUCT)%':
                pre_checkout_validator: App\Payments\PaymentsServicePreCheckoutQueryValidator

# Enable Behat testing support in test environment
when@test:
    avodel_telegram_bot_api:
        behat: true

framework:
    cache:
        pools:
            telegram_data_pool: null
```

### Step 3: Configure Cache for Thread/User Storage

```yaml
# config/services.yaml
Avodel\TelegramBotApi\Repository\KeyValueThreadRepository:
    $cache: '@telegram_data_pool'
    $threadClass: '%avodel.telegram_bot_api.thread_class%'
    $token: '%avodel.telegram_bot_api.token%'

Avodel\TelegramBotApi\Repository\KeyValueUserRepository:
    $cache: '@telegram_data_pool'
    $userClass: '%avodel.telegram_bot_api.user_class%'
    $token: '%avodel.telegram_bot_api.token%'
```

### Step 4: Configure Webhook Route

Create `config/routes/telegram_webhook.yaml`:

```yaml
telegram_webhook:
    path: /telegram/webhook
    controller: Avodel\TelegramBotApi\Controller\TelegramWebhookController
    methods: [POST]
```

### Step 5: Configure Environment Variables

Create or update your `.env` file with the required environment variables:

```bash
# Bot token from @BotFather (required)
TELEGRAM_BOT_API_TOKEN=your_bot_token_here

# Payment provider token from @BotFather (required for payments)
TELEGRAM_BOT_API_PAYMENT_PROVIDER_TOKEN=your_payment_provider_token

# Allowed IPs for webhook security (comma-separated)
TELEGRAM_BOT_API_ALLOWED_WEBHOOK_IPS="127.0.0.1, ::1"

# Telegram API base URL (default: https://api.telegram.org)
TELEGRAM_BASE_URL=https://api.telegram.org
```

#### Environment Variables Reference

| Variable | Required | Description |
|----------|----------|-------------|
| `TELEGRAM_BOT_API_TOKEN` | Yes | Bot token obtained from @BotFather |
| `TELEGRAM_BOT_API_PAYMENT_PROVIDER_TOKEN` | For payments | Payment provider token from @BotFather |
| `TELEGRAM_BOT_API_ALLOWED_WEBHOOK_IPS` | Recommended | Comma-separated list of allowed IPs for webhook |
| `TELEGRAM_BASE_URL` | No | Telegram API base URL (default: `https://api.telegram.org`) |

#### Webhook Security

For production, configure `TELEGRAM_BOT_API_ALLOWED_WEBHOOK_IPS` with Telegram's IP ranges:

```bash
TELEGRAM_BOT_API_ALLOWED_WEBHOOK_IPS="149.154.160.0/20, 91.108.4.0/22"
```

---

## Entities

### User Entity

Implement `Avodel\TelegramBotApi\Contract\UserInterface` from the bundle:

```php
<?php

declare(strict_types=1);

namespace App\Entity;

use Avodel\TelegramBotApi\Contract\UserInterface;
use DateTimeImmutable;

class User implements UserInterface
{
    private ?int $id = null;
    private bool $banned = false;
    private string $languageCode = 'en';
    private string $firstName = '';
    private string $lastName = '';
    private string $username = '';
    private ?string $textCallback = null;
    private ?string $lastInteractedThreadId = null;
    private ?DateTimeImmutable $createdAt = null;

    // Required by UserInterface
    public function getId(): int
    {
        return $this->id;
    }

    public function getLanguageCode(): string
    {
        return $this->languageCode;
    }

    public function getTextCallback(): ?string
    {
        return $this->textCallback;
    }

    public function getLastInteractedThreadId(): ?string
    {
        return $this->lastInteractedThreadId;
    }

    public function isBanned(): bool
    {
        return $this->banned;
    }

    // ... setters and other getters
}
```

### Thread Entity

Implement `Avodel\TelegramBotApi\Contract\ThreadInterface` for conversation state:

```php
<?php

declare(strict_types=1);

namespace App\Entity;

use Avodel\TelegramBotApi\Contract\ThreadInterface;

class Thread implements ThreadInterface
{
    private ?string $id = null;
    private array $state = [];
    private ?int $messageId = null;
    private array $queryCallBacks = [];
    private int $chatId;

    public function getId(): ?string
    {
        return $this->id;
    }

    public function getState(): array
    {
        return $this->state;
    }

    public function setStateItem(string $key, array $state): void
    {
        $this->state[$key] = $state;
    }

    public function setState(array $state): void
    {
        $this->state = $state;
    }

    public function unsetState(string $key): void
    {
        unset($this->state[$key]);
    }

    public function getMessageId(): ?int
    {
        return $this->messageId;
    }

    public function getQueryCallBacks(): array
    {
        return $this->queryCallBacks;
    }

    public function getChatId(): int
    {
        return $this->chatId;
    }

    // ... setters
}
```

---

## Callback Data Lifecycle

The bundle uses `Avodel\TelegramBotApi\CallbackQueryConvertor` to transform callback data when communicating with Telegram API. Understanding this transformation is essential for proper testing and debugging.

### Transformation Flow

```
┌─────────────────────────────────────────────────────────────────────────┐
│                        SENDING TO TELEGRAM                              │
├─────────────────────────────────────────────────────────────────────────┤
│  Your code                    →  Telegram API                           │
│  CallbackQueryButton("start_continue")                                  │
│                               →  button:start_continue                  │
│                                                                         │
│  WildcardCallbackQueryButton("select_item", "42")                       │
│                               →  wildcard:select_item:42                │
└─────────────────────────────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────────────────────────────┐
│                       RECEIVING FROM TELEGRAM                           │
├─────────────────────────────────────────────────────────────────────────┤
│  Telegram API                 →  Your handler                           │
│  button:start_continue        →  start_continue                         │
│  wildcard:select_item:42      →  select_item (with value "42")          │
└─────────────────────────────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────────────────────────────┐
│                       THREAD STORAGE                                    │
├─────────────────────────────────────────────────────────────────────────┤
│  thread.queryCallbacks stores callbacks WITHOUT prefix:                 │
│  ["start_continue", "start_deny", "select_item"]                        │
└─────────────────────────────────────────────────────────────────────────┘
```

### Callback Types

| Type | Format | Use Case |
|------|--------|----------|
| `button` | `button:<callback_id>` | Standard inline keyboard buttons |
| `wildcard` | `wildcard:<callback_id>:<value>` | Buttons with dynamic values (e.g., item selection) |

### Example Usage

```php
use Avodel\TelegramBotApi\CallbackQueryButton;
use Avodel\TelegramBotApi\WildcardCallbackQueryButton;
use Avodel\TelegramBotApi\Contract\ReceivedCallbackQueryMessageInterface;
use Avodel\TelegramBotApi\Contract\ThreadInteractorInterface;

// Creating a standard button
new CallbackQueryButton('accept', 'start_continue');
// Sent to Telegram as: button:start_continue

// Creating a wildcard button with dynamic value
new WildcardCallbackQueryButton('select', 'select_item', '42');
// Sent to Telegram as: wildcard:select_item:42

// In your handler, you receive the clean callback data
public function onCallbackQuery(ReceivedCallbackQueryMessageInterface $message, ThreadInteractorInterface $thread): void
{
    $callbackData = $message->getCallbackData(); // "start_continue" or "select_item"
    $value = $message->getValue(); // null or "42" for wildcard
}
```

---

## Handler Interfaces

The bundle provides four main handler interfaces:

| Interface | Purpose |
|-----------|---------|
| `Avodel\TelegramBotApi\Contract\CommandInterface` | Handle `/command` messages |
| `Avodel\TelegramBotApi\Contract\DialogInterface` | Manage dialog flow (entry points for conversations) |
| `Avodel\TelegramBotApi\Contract\TextMessageHandlerInterface` | Process text input from users |
| `Avodel\TelegramBotApi\Contract\CallbackQueryHandlerInterface` | Handle inline keyboard button presses |

### CommandInterface

Use for bot commands like `/start`, `/help`, `/new`:

```php
<?php

declare(strict_types=1);

namespace App\Dialog\Start;

use App\Dialog\Application\Request\NewRequestCommand;
use Avodel\TelegramBotApi\Contract\CommandInterface;
use Avodel\TelegramBotApi\Contract\ThreadInteractorInterface;

final readonly class StartCommand implements CommandInterface
{
    public function __construct(
        private NewRequestCommand $newRequestCommand,
    ) {
    }

    public function start(ThreadInteractorInterface $thread): void
    {
        // Delegate to another command/dialog
        $this->newRequestCommand->start($thread);
    }

    public function getSubscribedCallbacks(): array
    {
        return [];
    }

    public function getName(): string
    {
        return 'start'; // Responds to /start command
    }
}
```

---

## Callback Query Handlers

Combine `DialogInterface` with `CallbackQueryHandlerInterface` to handle inline keyboard button presses:

```php
use Avodel\TelegramBotApi\Contract\{CallbackQueryHandlerInterface, CommandInterface, MessageBuilder};

class NewRequestCommand implements CommandInterface, CallbackQueryHandlerInterface
{
    private const string TERMS_ACCEPT = '2b773991-1fa0-4530-9b6f-71a0ee4763f7';
    private const string TERMS_DECLINE = 'ee593f3e-d6ff-4a99-8164-fc5eb89b4332';

    public function start(ThreadInteractorInterface $thread): void
    {
        $thread->reply(
            MessageBuilder::create()
                ->text('terms.main')
                ->inlineKeyboardLine([
                    new CallbackQueryButton('decline', self::TERMS_DECLINE),
                    new CallbackQueryButton('accept', self::TERMS_ACCEPT),
                ])
                ->build(),
        );
    }

    public function getSubscribedCallbacks(): array
    {
        return [self::TERMS_ACCEPT, self::TERMS_DECLINE];
    }

    public function onCallbackQuery(
        ReceivedCallbackQueryMessageInterface $message,
        ThreadInteractorInterface $thread
    ): void {
        match ($message->getCallbackData()) {
            self::TERMS_ACCEPT => $this->nextDialog->start($thread),
            self::TERMS_DECLINE => $thread->reply(
                MessageBuilder::create()->text('terms.decline_message')->build()
            ),
        };
    }

    public function getName(): string
    {
        return 'new';
    }
}
```

---

## Text Message Handlers

Implement `TextMessageHandlerInterface` to handle free-form text input. Use `textCallBack()` in `MessageBuilder` to register for text input:

```php
use Avodel\TelegramBotApi\Contract\{DialogInterface, TextMessageHandlerInterface, MessageBuilder};

class FirstNameDialog implements DialogInterface, TextMessageHandlerInterface
{
    private ?string $firstNameCallback = null;

    public function start(ThreadInteractorInterface $thread): void
    {
        $thread->reply(
            MessageBuilder::create()
                ->text('Enter your first name')
                ->textCallBack($this->firstNameCallback) // Register for text input
                ->build(),
        );
    }

    public function onTextMessage(
        ReceivedTextMessageInterface $message,
        ThreadInteractorInterface $thread
    ): void {
        $firstName = $message->getText();

        if (!$this->validator->isNameValid($firstName)) {
            $thread->reply(
                MessageBuilder::create()->text('Invalid name')->build()
            );
            return;
        }

        // Save to state and proceed
        $state = $thread->readState(ApplicationRequestPropertiesState::class);
        $state->setFirstName($firstName);
        $thread->writeState($state);
        $this->nextDialog?->start($thread);
    }

    public function getSubscribedCallbacks(): array
    {
        return array_filter([$this->firstNameCallback]);
    }

    // Configure via setter injection (see Dialog Chains section)
    public function setFirstNameCallback(string $callback): void
    {
        $this->firstNameCallback = $callback;
    }
}
```

---

## State Management

### ThreadStateInterface

Create typed state objects for storing conversation data:

```php
<?php

declare(strict_types=1);

namespace App\Dialog\Application\RequestProperties;

use Avodel\TelegramBotApi\Contract\ThreadStateInterface;

class ApplicationRequestPropertiesState implements ThreadStateInterface
{
    private ?string $firstName = null;
    private ?string $surname = null;
    private ?string $birthday = null;
    private ?string $passportNumber = null;
    private ?string $citizenship = null;
    private ?string $phoneNumber = null;

    public static function getKey(): string
    {
        return 'application_request_properties';
    }

    public function serialize(): array
    {
        return [
            'firstName' => $this->firstName,
            'surname' => $this->surname,
            'birthday' => $this->birthday,
            'passportNumber' => $this->passportNumber,
            'citizenship' => $this->citizenship,
            'phoneNumber' => $this->phoneNumber,
        ];
    }

    public static function deserialize(array $data): self
    {
        $state = new self();
        $state->firstName = $data['firstName'] ?? null;
        $state->surname = $data['surname'] ?? null;
        $state->birthday = $data['birthday'] ?? null;
        $state->passportNumber = $data['passportNumber'] ?? null;
        $state->citizenship = $data['citizenship'] ?? null;
        $state->phoneNumber = $data['phoneNumber'] ?? null;

        return $state;
    }

    // Getters and setters
    public function getFirstName(): ?string
    {
        return $this->firstName;
    }

    public function setFirstName(?string $firstName): void
    {
        $this->firstName = $firstName;
    }

    // ... other getters/setters
}
```

### Using State in Handlers

```php
use App\Dialog\Application\RequestProperties\ApplicationRequestPropertiesState;

// Read state
$state = $thread->readState(ApplicationRequestPropertiesState::class);

// Modify state
$state->setFirstName($firstName);

// Write state back
$thread->writeState($state);

// Clear state
$thread->unsetState(ApplicationRequestPropertiesState::class);
```

---

## Dialog Chains

### Service Configuration Pattern

Configure dialog chains using setter injection in `config/services.yaml`:

```yaml
services:
    _defaults:
        autowire: true
        autoconfigure: true

    app.fill_firstname_dialog:
        class: App\Dialog\Application\RequestProperties\FirstNameDialog
        calls:
            # Link to next dialog in chain
            - setNextDialog: ['@app.fill_surname_dialog']
            # Configure back button with previous dialog reference and callback UUID
            - setPreviousCallback: ['@app.fill_mos_code_dialog', 'c12a2023-d19d-487d-a003-0ae31f53a73b']
            # Register text callback UUID for this dialog
            - setFirstNameCallback: ['a6fbd108-8925-4946-8925-b8479dc6f2bf']

    app.fill_surname_dialog:
        class: App\Dialog\Application\RequestProperties\SurnameDialog
        calls:
            - setNextDialog: ['@app.fill_dob_dialog']
            - setPreviousCallback: ['@app.fill_firstname_dialog', '4d8c27da-0752-403f-8a49-e65de2d3ea64']
            - setSurnameCallback: ['74ce2cbb-7233-421c-96d1-9c69c6e83c24']

    app.fill_dob_dialog:
        class: App\Dialog\Application\RequestProperties\DOBDialog
        calls:
            - setNextDialog: ['@app.fill_passport_number_dialog']
            - setPreviousCallback: ['@app.fill_surname_dialog', '58a7bf84-cdeb-4bbc-a3fc-7dd564ee8a11']
            - setDOBCallback: ['c12808fb-871e-420e-8165-07b5e0819abf']
```

---

## Behat Testing Setup

```yaml
# config/packages/avodel_telegram_bot_api.yaml
when@test:
    avodel_telegram_bot_api:
        behat: true
```

### Test Structure

Each scenario follows a three-part structure:

1. **Given** — Set up precondition state (user, thread, callbacks)
2. **When** — User action (message, command, button click)
3. **Then** — Assert result (message text, keyboard, state)

**Important:** Each scenario must be isolated. Set up state directly rather than executing previous commands.

---

## Behat Test Examples

### Command with Keyboard

```gherkin
Scenario: Start command shows welcome with button
  Given there is a telegram user
  When this telegram user sends "/start" message
  Then this telegram user should receive this message:
    """
    Welcome! How can I help?
    """
  And this message should have the following keyboard:
    | Text  | Callback Data                        |
    | Start | a1b2c3d4-e5f6-7890-abcd-ef1234567890 |
```

### Callback Query

```gherkin
Scenario: Button click triggers next step
  Given there is a telegram user
  And there is a thread for this telegram user
  And this thread has the following callbacks:
    | Callback Data                        |
    | a1b2c3d4-e5f6-7890-abcd-ef1234567890 |
  When this telegram user sends "button:a1b2c3d4-e5f6-7890-abcd-ef1234567890" callback from this thread
  Then this telegram user should receive this message:
    """
    What is your name?
    """
```

### Text Input with State

```gherkin
Scenario: Text input saves to state
  Given there is a telegram user
  And there is a thread for this telegram user
  And this telegram user has "name-callback-uuid" text callback
  When this telegram user sends "John" message
  Then this thread should have the following state for "user_data" key:
    """
    {"firstName":"John"}
    """
```

### Validation Error

```gherkin
Scenario: Invalid input shows error
  Given this telegram user has "name-callback-uuid" text callback
  When this telegram user sends "12345" message
  Then this telegram user should receive this message:
    """
    Invalid name.
    """
```

### HTTP Stub

```gherkin
Scenario: External API call
  Given there is a stubbed http interaction:
  """
    >>>
    GET http://api/endpoint

    <<<
    HTTP/2 200 OK
  """
  When this telegram user sends "/start" message
  Then this telegram user should receive this message:
    """
    Success
    """
```

---

## Behat Step Reference

The bundle provides `TelegramContext` with these step definitions:

### Setup Steps

| Step | Description |
|------|-------------|
| `Given there is a telegram user` | Creates user with random ID |
| `Given there is a telegram user with id {id}` | Creates user with specific ID |
| `Given this telegram user has locale "{locale}"` | Sets language code |
| `Given this telegram user has "{callback}" text callback` | Sets text callback |
| `Given there is a thread for this telegram user` | Creates empty thread |
| `Given there is a thread for this telegram user with the following state:` | Creates thread with JSON state |
| `Given this thread has the following callbacks:` | Sets allowed callbacks (table) |

### Action Steps

| Step | Description |
|------|-------------|
| `When this telegram user sends "{text}" message` | Sends text/command |
| `When this telegram user sends "{callback}" callback from this thread` | Sends callback with prefix |

### Assertion Steps

| Step | Description |
|------|-------------|
| `Then this telegram user should receive this message:` | Asserts message text |
| `And this message should have the following keyboard:` | Asserts keyboard buttons |
| `And this thread should have the following state for "{key}" key:` | Asserts state key |
| `Then the successful answer is sent for "{id}" pre checkout query` | Payment success |

### Important: Callback Format

When **sending** callbacks, use prefix: `button:uuid` or `wildcard:id:value`

When **registering** callbacks in thread, store **without** prefix.

### Important: Table Format

First row is always a header:

```gherkin
Given this thread has the following callbacks:
  | Callback Data |   ← Header (required)
  | uuid-here     |   ← Data row
```

---

## Custom Behat Context

Use `TelegramApiStub` to access sent messages in custom contexts:

```php
use Avodel\TelegramBotApi\Behat\TelegramApiStub;

final readonly class NotificationContext implements Context
{
    public function __construct(
        private TelegramApiStub $telegramApiStub,
    ) {}

    /**
     * @Then /^the user (\d+) should receive the following message$/
     */
    public function theUserShouldReceiveMessage(int $id, PyStringNode $text): void
    {
        $messages = $this->telegramApiStub->getSentMessages()[$id] ?? [];
        foreach ($messages as $msg) {
            if (trim($msg->getText()) === trim($text->getRaw())) {
                return;
            }
        }
        throw new \RuntimeException('Expected message was not sent');
    }
}
```

---

## Best Practices

### 1. Use UUID-based Callback Identifiers

Always use UUIDs for callback data to ensure uniqueness and prevent collisions:

```php
private const string ACCEPT_BUTTON = '2b773991-1fa0-4530-9b6f-71a0ee4763f7';
private const string DECLINE_BUTTON = 'ee593f3e-d6ff-4a99-8164-fc5eb89b4332';
```

### 2. Implement Validation in Handlers

Always validate user input before saving to state:

```php
use Avodel\TelegramBotApi\Contract\MessageBuilder;
use Avodel\TelegramBotApi\Contract\ReceivedTextMessageInterface;
use Avodel\TelegramBotApi\Contract\ThreadInteractorInterface;

public function onTextMessage(ReceivedTextMessageInterface $message, ThreadInteractorInterface $thread): void
{
    $input = $message->getText();

    if (!$this->validator->isValid($input)) {
        $thread->reply(
            MessageBuilder::create()
                ->text('validation.error')
                ->build(),
        );
        return;
    }

    // Process valid input
}
```

### 3. Use Typed State Objects

Create dedicated state classes instead of using raw arrays:

```php
use App\Dialog\Application\RequestProperties\ApplicationRequestPropertiesState;

// Good
$state = $thread->readState(ApplicationRequestPropertiesState::class);
$state->setFirstName($firstName);
$thread->writeState($state);

// Avoid
$thread->setState(['firstName' => $firstName]);
```

### 4. Separate Concerns

- **Commands** - Entry points for user-initiated actions
- **Dialogs** - Multi-step conversation flows
- **State classes** - Typed data containers
- **Validators** - Input validation logic
- **Providers** - External service integration

### 5. Handle Errors Gracefully

Always catch and handle exceptions from external services:

```php
use App\Exception\ApplicationRequestsLimitReachedException;

try {
    $this->applicationRequestApi->create($request);
} catch (ApplicationRequestsLimitReachedException) {
    $this->limitReachedDialog->start($thread);
    return;
}
```

### 6. Use MessageBuilder for Responses

The `Avodel\TelegramBotApi\Contract\MessageBuilder` provides a fluent API for constructing messages:

```php
use Avodel\TelegramBotApi\CallbackQueryButton;
use Avodel\TelegramBotApi\Contract\MessageBuilder;

$thread->reply(
    MessageBuilder::create()
        ->markdown()  // Enable markdown formatting
        ->text('message.key')  // Translation key
        ->textCallBack($callbackId)  // Register for text input
        ->inlineKeyboardLine([
            new CallbackQueryButton('button.text', $callbackUuid),
        ])
        ->build(),
);
```

### 8. Test HTTP Interactions

Stub external HTTP calls in Behat tests:

```gherkin
And there is a stubbed http interaction:
"""
  >>>
  POST http://api-service/endpoint

  <<<
  HTTP/2 200 OK
  Content-Type: application/json

  {"id": 1, "status": "created"}
"""
```

---

## Summary

The `avodel/telegram-bot-api-bundle` provides a structured approach to building Telegram bots:

1. **Configure** the bundle with your entities, repositories, and services
2. **Implement handlers** using the appropriate interfaces (Command, Dialog, TextMessage, CallbackQuery)
3. **Manage state** using typed state objects implementing `Avodel\TelegramBotApi\Contract\ThreadStateInterface`
4. **Chain dialogs** together using service configuration with setter injection
5. **Test** using Behat with the bundle's testing support and stubs

This architecture promotes clean separation of concerns, testability, and maintainability for complex Telegram bot applications.
