<?php

declare(strict_types=1);

namespace Avodel\TelegramBotApi\Async;

use Avodel\TelegramBotApi\Async\Messenger\EditMessage;
use Avodel\TelegramBotApi\Async\Messenger\SendInvoiceMessage;
use Avodel\TelegramBotApi\Async\Messenger\SendMessage;
use Avodel\TelegramBotApi\Contract\IssueInvoiceInterface;
use Avodel\TelegramBotApi\Contract\MessageInterface;
use Avodel\TelegramBotApi\Contract\PhotoInterface;
use Avodel\TelegramBotApi\Contract\SentMessageInterface;
use Avodel\TelegramBotApi\Contract\TelegramApiInterface;
use BadMethodCallException;
use Symfony\Component\Messenger\MessageBusInterface;

final class AsyncTelegramApi implements TelegramApiInterface
{
    public function __construct(
        private readonly MessageBusInterface $messageBus,
    )
    {
    }

    public function sendMessage(int $chatId, MessageInterface $message): SentMessageInterface
    {
        $this->messageBus->dispatch(new SendMessage($chatId, $message));

        return new AsyncSentMessage();
    }

    public function editMessage(int $chatId, int $messageId, MessageInterface $message): void
    {
        $this->messageBus->dispatch(new EditMessage($chatId, $messageId, $message));
    }

    public function sendInvoice(int $chatId, IssueInvoiceInterface $invoice): void
    {
        $this->messageBus->dispatch(new SendInvoiceMessage($chatId, $invoice));
    }

    public function answerPreCheckoutQuery(string $preCheckoutQueryId, bool $success): void
    {
        throw new BadMethodCallException('The answerPreCheckoutQuery method is not supported for async.');
    }

    public function getUpdates(int $offset = 0): array
    {
        throw new BadMethodCallException('The getUpdates method is not supported for async.');
    }

    public function sendPhoto(int $chatId, PhotoInterface $photo): void
    {
        throw new BadMethodCallException('The sendPhoto method is not supported for async.');
    }

    public function getChatMember(int $chatId, int $userId): array
    {
        throw new BadMethodCallException('The getChatMember method is not supported for async.');
    }

    public function deleteMessage(int $chatId, int $messageId): void
    {
        throw new BadMethodCallException('The deleteMessage method is not supported for async.');
    }
}
