<?php

declare(strict_types=1);

namespace Avodel\TelegramBotApi\Behat;

use Avodel\TelegramBotApi\Contract\MessageInterface;
use Avodel\TelegramBotApi\Contract\ThreadRepositoryInterface;
use Avodel\TelegramBotApi\Contract\UserInterface;
use Avodel\TelegramBotApi\Contract\UserRepositoryInterface;
use Avodel\TelegramBotApi\ThreadFactory;
use Avodel\TelegramBotApi\UserFactory;
use Behat\Behat\Context\Context;
use Behat\Gherkin\Node\PyStringNode;
use Behat\Gherkin\Node\TableNode;
use RuntimeException;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\KernelInterface;
use Webmozart\Assert\Assert;

class TelegramContext implements Context
{
    private ?MessageInterface $message = null;

    private ?int $telegramUserId = null;

    private ?string $threadId = null;

    public function __construct(
        private readonly TelegramApiStub $telegramApiStub,
        private readonly KernelInterface $kernel,
        private readonly ThreadFactory $threadFactory,
        private readonly UserFactory $userFactory,
        private readonly UserRepositoryInterface $userRepository,
        private readonly ThreadRepositoryInterface $threadRepository,
    )
    {
    }


    /**
     * @Transform /^(?:this|that|the) (telegram user)$/
     */
    public function getTelegramUserId(): int
    {
        if (!$this->telegramUserId) {
            throw new RuntimeException('Telegram user is not set.');
        }

        return $this->telegramUserId;
    }

    /**
     * @Transform /^(?:this|that|the) (message)$/
     */
    public function getMessage(): MessageInterface
    {
        if (!$this->message) {
            throw new RuntimeException('Telegram message is not set.');
        }

        return $this->message;
    }

    /**
     * @Transform /^(?:this|that|the) (thread)$/
     */
    public function getThreadId(): string
    {
        if (!$this->threadId) {
            throw new RuntimeException('Thread is not set.');
        }

        return $this->threadId;
    }

    /**
     * @Given /^there is a telegram user$/
     * @Given /^there is a telegram user with id ([^"]*)$/
     */
    public function thereIsATelegramUser(int $userId = null): void
    {
        $user = $this->userFactory->create(
            [
                'last_name' => 'Test Lastname',
                'id' => $userId ?? random_int(10000, 2000000),
                'first_name' => 'Test',
                'username' => 'Test',
                'language_code' =>  'en'
            ]
        );

        $this->userRepository->save($user);
        $this->telegramUserId = $user->getId();
    }

    /**
     * @Given /^(this telegram user) has locale "([^"]*)"$/
     */
    public function thisTelegramUserHasLocale(int $telegramUserId, string $locale): void
    {
        $user = $this->userRepository->get($telegramUserId);
        $user->setLanguageCode($locale);
        $this->userRepository->save($user);
    }

    /**
     * @Given /^(this telegram user) has "([^"]*)" text callback$/
     */
    public function thisTelegramUserHasTextCallback(int $telegramUserId, string $textCallback): void
    {
        $user = $this->userRepository->get($telegramUserId);
        $user->setTextCallback($textCallback);
        $this->userRepository->save($user);
    }

    /**
     * @When /^(this telegram user) sends "([^"]*)" message$/
     */
    public function thisTelegramUserSendsMessage(int $telegramUserId, string $message): void
    {
        $user = $this->userRepository->get($telegramUserId);

        $update = [
            'update_id' => 10000,
            'message' => [
                'date' => 1441645532,
                'chat' => [
                    'last_name' => $user->getLastName(),
                    'id' => $user->getId(),
                    'first_name' => $user->getFirstName(),
                    'username' => $user->getUsername(),
                    'type' => 'private'
                ],
                'message_id' => 75,
                'from' => [
                    'last_name' => $user->getLastName(),
                    'id' => $user->getId(),
                    'first_name' => $user->getFirstName(),
                    'username' => $user->getLastName(),
                    'language_code' => $user->getLanguageCode()
                ],
                'text' => $message
            ]
        ];

        $this->sendUpdate($update);
    }

    private function sendUpdate(array $update): void
    {
        $response = $this->kernel->handle(
            Request::create('/telegram/webhook', 'POST', [], [], [], ['CONTENT_TYPE' => 'application/json'], json_encode($update, JSON_THROW_ON_ERROR))
        );

        if ($response->getStatusCode() > 299) {
            throw new RuntimeException($response->getContent());
        }
    }

    /**
     * @Then /^(this telegram user) should receive this message:$/
     */
    public function thisTelegramUserShouldReceiveThisMessage(int $telegramUserId, PyStringNode $string): void
    {
        $sentMessages = $this->telegramApiStub->getSentMessages()[$telegramUserId] ?? [];

        foreach ($sentMessages as $sentMessage) {
            if (trim($sentMessage->getText()) === trim($string->getRaw())) {
                $this->message = $sentMessage;

                return;
            }
        }

        $allMessages = array_map(fn(MessageInterface $message) => $message->getText(), $sentMessages);

        throw new RuntimeException(
            'The telegram message was not sent.' . PHP_EOL . PHP_EOL . PHP_EOL .
            'Expected message:' . PHP_EOL . $string->getRaw() . PHP_EOL .
            'Got messages:' . PHP_EOL . implode('; ', $allMessages)
        );
    }

    /**
     * @Given /^(this message) should have the following keyboard:$/
     */
    public function thisMessageShouldHaveTheFollowingKeyboard(MessageInterface $message, TableNode $table): void
    {
        Assert::notEmpty($this->message->getInlineKeyboardLines(), 'Inline keyboard is not found.');
        $inlineKeyboardButtons = [];

        foreach ($message->getInlineKeyboardLines() as $inlineKeyboardLines) {
            foreach ($inlineKeyboardLines as $inlineKeyboard) {
                $inlineKeyboardButtons[] = ['text' => $inlineKeyboard->getText(), 'callbackData' => $inlineKeyboard->getCallbackData()];
            }
        }

        $expectedButtons = [];

        foreach ($table as $row) {
            $text = $row['Text'];
            if (str_starts_with($text, '"') && str_ends_with($text, '"')) {
                $text = substr($text, 1, -1);
            }

            $expectedButtons[] = ['text' => $text, 'callbackData' => $row['Callback Data']];
        }

        Assert::eq(
            $expectedButtons,
            $inlineKeyboardButtons,
            $this->getButtonsDiffMessage($expectedButtons, $inlineKeyboardButtons)
        );
    }

    private function getButtonsDiffMessage(array $expectedButtons, array $givenButton): string
    {
        $string = 'Expected following buttons: ' . PHP_EOL;

        foreach ($expectedButtons as $button) {
            $string .= $button['text'] . ' => ' . $button['callbackData'] . PHP_EOL;
        }

        $string .= PHP_EOL . 'Got following buttons: ' . PHP_EOL;

        foreach ($givenButton as $button) {
            $string .= $button['text'] . ' => ' . $button['callbackData'] . PHP_EOL;
        }

        return $string;
    }

    /**
     * @When /^(this telegram user) sends "([^"]*)" callback$/
     */
    public function thisTelegramUserSendsCallback(UserInterface $user, string $callbackData): void
    {

    }

    /**
     * @Given /^there is a thread for (this telegram user)$/
     * @Given /^there is a thread for (this telegram user) with the following state:$/
     */
    public function thereIsAThreadForThisTelegramUser(int $telegramUserId, PyStringNode $state = null): void
    {
        $thread = $this->threadFactory->createEmptyWithChatId($telegramUserId);
        $thread->setMessageId(random_int(10000, 200000));
        if ($state) {
            $stateData = json_decode($state->getRaw(), true, 512, JSON_THROW_ON_ERROR);
            foreach ($stateData as $key => $value) {
                $thread->setStateItem($key, $value);
            }
        }
        $this->threadRepository->save($thread);
        $this->threadId = $thread->getId();

        $user = $this->userRepository->get($telegramUserId);
        $user->setLastInteractedThreadId($this->threadId);
        $this->userRepository->save($user);
    }

    /**
     * @Given /^(this thread) has the following callbacks:$/
     */
    public function thisThreadHasTheFollowingCallbacks(string $threadId, TableNode $table): void
    {
        $thread = $this->threadRepository->findOneById($threadId);
        Assert::notNull($thread);

        $callbacks = [];

        foreach ($table as $row) {
            $callbacks[] = reset($row);
        }

        $thread->setQueryCallBacks($callbacks);
        $this->threadRepository->save($thread);
    }

    /**
     * @When /^(this telegram user) sends "([^"]*)" callback from (this thread)$/
     */
    public function thisTelegramUserSendsCallbackFromThisThread(int $telegramUserId, string $callbackData, string $threadId): void
    {
        $thread = $this->threadRepository->findOneById($threadId);
        Assert::notNull($thread);
        $user = $this->userRepository->get($telegramUserId);

        $update = [
            'update_id' => 10000,
            'callback_query' => [
                'id' => '4382bfdwdsb323b2d9',
                'message' => [
                    'message_id' => $thread->getMessageId(),
                    'chat' => [
                        'type' => 'private'
                    ]
                ],
                'from' => [
                    'last_name' => $user->getLastName(),
                    'type' => 'private',
                    'id' => $user->getId(),
                    'first_name' => $user->getFirstName(),
                    'username' => $user->getUsername(),
                    'language_code' => $user->getLanguageCode()
                ],
                'data' => $callbackData,
                'inline_message_id' => '1234csdbsk4839'
            ]
        ];

        $this->sendUpdate($update);
    }

    /**
     * @Given /^(this thread) has the following state:$/
     */
    public function thisThreadHasTheFollowingState(string $threadId, PyStringNode $string): void
    {
        $thread = $this->threadRepository->findOneById($threadId);
        Assert::notNull($thread);

        $states = json_decode($string->getRaw(), true, 512, JSON_THROW_ON_ERROR);

        foreach ($states as $key => $value) {
            $thread->setStateItem($key, $value);
        }

        $this->threadRepository->save($thread);
    }

    /**
     * @Given /^(this thread) should have the following state:$/
     */
    public function thisThreadShouldHasTheFollowingState(string $threadId, PyStringNode $string): void
    {
        $thread = $this->threadRepository->findOneById($threadId);
        Assert::notNull($thread);

        $states = json_decode($string->getRaw(), true, 512, JSON_THROW_ON_ERROR);
        Assert::eq($thread->getState(), $states);
    }

    /**
     * @Given /^(this thread) should have the following state for "([^"]*)" key:$/
     */
    public function thisThreadShouldHaveTheFollowingStateForKey(string $threadId, string $key, PyStringNode $string): void
    {
        $thread = $this->threadRepository->findOneById($threadId);
        Assert::notNull($thread);

        $state = json_decode($string->getRaw(), true, 512, JSON_THROW_ON_ERROR);
        Assert::keyExists($thread->getState(), $key, 'State key not found');

        Assert::eq($thread->getState()[$key], $state);
    }

    /**
     * @Given /^the thread with message ID "([^"]*)" should have the following state for the key "([^"]*)":$/
     */
    public function threadWithMessageIdShouldHaveTheFollowingStateForKey(int $messageId, string $key, PyStringNode $string): void
    {
        $thread = $this->threadRepository->getByMessageId($messageId);
        Assert::notNull($thread);

        $state = json_decode($string->getRaw(), true, 512, JSON_THROW_ON_ERROR);
        Assert::keyExists($thread->getState(), $key, 'State key not found');

        Assert::eq($thread->getState()[$key], $state);
    }

    /**
     * @Given /^the received Telegram message has the ID "([^"]*)"$/
     */
    public function receivedTelegramMessageHasId(int $messageId): void
    {
        $this->telegramApiStub->setReceivedMessageId($messageId);
    }

    /**
     * @Then /^the (successful|unsuccessful) answer is sent for "([^"]*)" pre checkout query$/
     */
    public function theBotShouldAnswerOnPreCheckoutQuery(string $status, string $preCheckoutQueryId): void
    {
        $preCheckoutQueries = $this->telegramApiStub->getSentAnswerPreCheckoutQueries();
        Assert::keyExists(
            $preCheckoutQueries,
            $preCheckoutQueryId,
            sprintf('There is no answer for pre-checkout-query %d.', $preCheckoutQueryId)
        );
        Assert::eq(
            $status === 'successful',
            $preCheckoutQueries[$preCheckoutQueryId],
            'Pre-checkout-query answer in invalid state.'
        );
    }
}
