<?php

declare(strict_types=1);

namespace Avodel\TelegramBotApi\Repository;

use Avodel\TelegramBotApi\Contract\UserInterface;
use Avodel\TelegramBotApi\Contract\UserRepositoryInterface;
use Avodel\TelegramBotApi\Exception\UserNotFoundException as TelegramUserNotFoundException;
use Symfony\Component\Serializer\SerializerInterface;
use Symfony\Contracts\Cache\CacheInterface;

final class KeyValueUserRepository implements UserRepositoryInterface
{
    public function __construct(
        private readonly CacheInterface $cache,
        private readonly SerializerInterface $serializer,
        private readonly string $token,
        private readonly string $userClass,
    )
    {
    }

    private function getKey(int $userId): string
    {
        [$botId, ] = explode(':', $this->token);

        return 'user_' . $botId . '_' . $userId;
    }

    public function get(int $userId): UserInterface
    {
        /** @var string|null $data */
        $data = $this->cache->get($this->getKey($userId), fn() => null);

        if (!$data) {
            throw new TelegramUserNotFoundException('User not found. ID: ' . $userId);
        }

        return $this->serializer->deserialize($data, $this->userClass, 'json');
    }

    public function save(UserInterface $user): void
    {
        $key = $this->getKey($user->getId());
        $serializedUser = $this->serializer->serialize($user, 'json');

        $this->cache->delete($key);
        $this->cache->get($key, fn() => $serializedUser);
    }
}
