<?php

declare(strict_types=1);

namespace Avodel\TelegramBotApi\UpdateHandler;

use Avodel\TelegramBotApi\CallbackQueryConvertor;
use Avodel\TelegramBotApi\CallbackQueryHandlerRegistry;
use Avodel\TelegramBotApi\Contract\ExceptionHandlerInterface;
use Avodel\TelegramBotApi\Contract\ThreadInteractorInterface;
use Avodel\TelegramBotApi\Contract\ThreadInterface;
use Avodel\TelegramBotApi\Contract\ThreadRepositoryInterface;
use Avodel\TelegramBotApi\Contract\UpdateHandlerInterface;
use Avodel\TelegramBotApi\Contract\UserNotFoundExceptionInterface;
use Avodel\TelegramBotApi\Contract\UserRepositoryInterface;
use Avodel\TelegramBotApi\Exception\CallbackHandlerNotFoundException;
use Avodel\TelegramBotApi\Exception\CallbackQueryConvertingException;
use Avodel\TelegramBotApi\ReceivedCallbackQueryMessage;
use Avodel\TelegramBotApi\ThreadInterfactorFactory;
use Avodel\TelegramBotApi\UserFactory;
use Psr\Log\LoggerInterface;
use Throwable;

final class CallbackQueryUpdateHandler implements UpdateHandlerInterface
{
    public function __construct(
        private readonly UserFactory $userFactory,
        private readonly LoggerInterface $logger,
        private readonly ExceptionHandlerInterface $exceptionHandler,
        private readonly UserRepositoryInterface $userRepository,
        private readonly CallbackQueryHandlerRegistry $callbackQueryHandlerRegistry,
        private readonly ThreadRepositoryInterface $threadRepository,
        private readonly ThreadInterfactorFactory $threadInterfactorFactory,
        private readonly CallbackQueryConvertor $callbackQueryConvertor,
    )
    {
    }

    public function handle(array $update): void
    {
        $callbackQuery = $update['callback_query'];

        $from = $callbackQuery['from'];
        $messageId = $callbackQuery['message']['message_id'];
        $rawCallBackQuery = $callbackQuery['data'];

        try {
            $receivedMessage = $this->callbackQueryConvertor->convertCallbackDataToReceivedMessage($messageId, $rawCallBackQuery);
        } catch (CallbackQueryConvertingException $e) {
            $this->logger->warning('Failed to convert callback query data', ['exception' => $e]);
            return;
        }

        try {
            $user = $this->userRepository->get($from['id']);
        } catch (UserNotFoundExceptionInterface $e) {
            $this->logger->info('User not found. New one will be created.', ['user_id' => $from['id'], 'exception' => $e]);
            $user = $this->userFactory->create($from);
            $this->userRepository->save($user);
        }

        $thread = $this->threadRepository->getByMessageId($messageId);
        $threadInteractor = $this->threadInterfactorFactory->create($user, $thread);

        try {
            $this->doHandle($receivedMessage, $threadInteractor, $thread);
        } catch (Throwable $throwable) {
            $this->exceptionHandler->handle($update, $user->getId(), $throwable);
        }

        $this->threadRepository->save($thread);
        $user->setLastInteractedThreadId($thread->getId());
        $this->userRepository->save($user);
    }

    private function doHandle(ReceivedCallbackQueryMessage $message, ThreadInteractorInterface $threadInteractor, ThreadInterface $thread): void
    {
        $callbackData = $message->getCallbackData();
        $queryCallbacks = $thread->getQueryCallBacks();

        if (!in_array($callbackData, $queryCallbacks, true)) {
            $this->logger->warning('Got a callback data which is not registered in user context.', [
                'callbackData' => $callbackData,
                'allCallbackData' => $queryCallbacks,
            ]);

            return;
        }

        try {
            $handler = $this->callbackQueryHandlerRegistry->getHandler($callbackData);
        } catch (CallbackHandlerNotFoundException) {
            $this->logger->error('Unable to find a callback handler for button.', [
                'callbackData' => $callbackData,
            ]);

            return;
        }

        $handler->onCallbackQuery($message, $threadInteractor);
    }

    public function supports(array $update): bool
    {
        $callbackQuery = $update['callback_query'] ?? null;

        if (!$callbackQuery) {
            return false;
        }

        if (($callbackQuery['message']['chat']['type'] ?? null) !== 'private') {
            return false;
        }

        return true;
    }
}
