<?php

declare(strict_types=1);

namespace Avodel\TelegramBotApi\UpdateHandler;

use Avodel\TelegramBotApi\Contract\InvoiceInterface;
use Avodel\TelegramBotApi\Contract\InvoiceNotFoundExceptionInterface;
use Avodel\TelegramBotApi\Contract\InvoiceRepositoryInterface;
use Avodel\TelegramBotApi\Contract\PreCheckoutQueryInvalidExceptionInterface;
use Avodel\TelegramBotApi\Contract\PreCheckoutValidatorInterface;
use Avodel\TelegramBotApi\Contract\TelegramApiInterface;
use Avodel\TelegramBotApi\Contract\UpdateHandlerInterface;
use Avodel\TelegramBotApi\Exception\PreCheckoutQueryInvalidException;
use Avodel\TelegramBotApi\PreCheckoutQuery;
use Avodel\TelegramBotApi\PreCheckoutValidatorsRegistry;
use Psr\Log\LoggerInterface;

final class PreCheckoutQueryUpdateHandler implements UpdateHandlerInterface
{
    public function __construct(
        private readonly PreCheckoutValidatorsRegistry $preCheckoutValidatorsRegistry,
        private readonly InvoiceRepositoryInterface $invoiceRepository,
        private readonly LoggerInterface $logger,
        private readonly TelegramApiInterface $telegramApi,
        private readonly bool $enableBasicPreCheckoutValidation,
    )
    {
    }

    public function handle(array $update): void
    {
        $preCheckout = PreCheckoutQuery::createFromPreCheckoutQueryUpdate($update['pre_checkout_query']);
        $validationPassed = false;

        try {
            $invoice = $this->invoiceRepository->get($preCheckout->getInvoicePayload());

            if ($this->enableBasicPreCheckoutValidation) {
                $this->performBasicValidation($preCheckout, $invoice);
            }

            $validator = $this->preCheckoutValidatorsRegistry->getValidator($invoice->getProduct());
            if ($validator instanceof PreCheckoutValidatorInterface) {
                $validator->validate($preCheckout);
            }

            $validationPassed = true;
        } catch (InvoiceNotFoundExceptionInterface $exception) {
            $this->logger->alert('Invoice not found while processing pre checkout', ['exception' => $exception, 'payload' => $preCheckout->getInvoicePayload()]);
        } catch (PreCheckoutQueryInvalidExceptionInterface $exception) {
            $this->logger->alert('Pre checkout validation error', ['exception' => $exception, 'payload' => $preCheckout->getInvoicePayload()]);
        }

        $this->telegramApi->answerPreCheckoutQuery($preCheckout->getPreCheckoutQueryId(), $validationPassed);
    }

    private function performBasicValidation(PreCheckoutQuery $preCheckoutQuery, InvoiceInterface $invoice): void
    {
        if ($preCheckoutQuery->getTotalAmount() < $invoice->getPricingAmount()) {
            throw new PreCheckoutQueryInvalidException('Total amount is not equal to price amount');
        }

        if ($preCheckoutQuery->getCurrency() !== $invoice->getPricingCurrency()) {
            throw new PreCheckoutQueryInvalidException('Currency is not equal to price currency');
        }

        if ($invoice->isPaid()) {
            throw new PreCheckoutQueryInvalidException('Invoice is already paid');
        }
    }


    public function supports(array $update): bool
    {
        return isset($update['pre_checkout_query']);
    }
}
