<?php

declare(strict_types=1);

namespace Avodel\TelegramBotApi\UpdateHandler;

use Avodel\TelegramBotApi\Contract\InvoiceNotFoundExceptionInterface;
use Avodel\TelegramBotApi\Contract\InvoiceRepositoryInterface;
use Avodel\TelegramBotApi\Contract\PostPaymentProcessingExceptionInterface;
use Avodel\TelegramBotApi\Contract\PostPaymentProcessorInterface;
use Avodel\TelegramBotApi\Contract\UpdateHandlerInterface;
use Avodel\TelegramBotApi\PostPaymentProcessorsRegistry;
use Psr\Clock\ClockInterface;
use Psr\Log\LoggerInterface;

final class SuccessfulPaymentHandler implements UpdateHandlerInterface
{
    public function __construct(
        private readonly PostPaymentProcessorsRegistry $postPaymentProcessorsRegistry,
        private readonly InvoiceRepositoryInterface    $invoiceRepository,
        private readonly ClockInterface                $clock,
        private readonly LoggerInterface               $logger,
    )
    {
    }

    /**
     * @throws PostPaymentProcessingExceptionInterface
     */
    public function handle(array $update): void
    {
        $payload = $update['message']['successful_payment']['invoice_payload'];

        try {
            $invoice = $this->invoiceRepository->get($payload);
        } catch (InvoiceNotFoundExceptionInterface $e) {
            $this->logger->alert('Invoice not found. While processing successful payment', ['payload' => $payload, 'exception' => $e]);
            return;
        }

        if (!$invoice->isPaid()) {
            $invoice->setPaid(true);
            $invoice->setPaidAt($this->clock->now());
            $this->invoiceRepository->save($invoice);
        } else {
            $this->logger->alert('Invoice already paid. Postprocessing without paying', ['payload' => $payload]);
        }

        $processor = $this->postPaymentProcessorsRegistry->getProcessor($invoice->getProduct());

        if (!$processor instanceof PostPaymentProcessorInterface) {
            $this->logger->info('No configured post payment processor', ['product' => $invoice->getProduct()]);
            return;
        }

        try {
            $processor->process($invoice);
        } catch (PostPaymentProcessingExceptionInterface $exception) {
            $this->logger->error('Post payment processing was failed, but invoice was marked as paid successfully.', ['exception' => $exception, 'payload' => $payload]);
        }
    }


    public function supports(array $update): bool
    {
        return isset($update['message']['successful_payment']);
    }
}
