# Avodel UUID

A lightweight PHP library that provides an injectable, testable interface for UUID generation. It wraps `ramsey/uuid` to enable dependency injection and deterministic testing in applications that need unique identifiers.

The library solves a common testing problem: when your services generate UUIDs internally, tests become non-deterministic. By depending on `UuidGeneratorInterface` instead of calling `Uuid::uuid4()` directly, you can inject a stub in tests to control generated values.

## Business Logic

### Core Concept

**Problem:** Services that generate UUIDs directly are hard to test. You cannot predict or assert on the generated identifiers.

**Solution:** Wrap UUID generation behind an interface. In production, use the real generator. In tests, inject a stub that returns predetermined values.

### Generation Flow

```
1. SERVICE NEEDS UUID
   │
   └── Calls UuidGeneratorInterface::generate()
           │
           ▼
2. IMPLEMENTATION SELECTION (via DI container)
   │
   ├── Production → UuidGenerator
   │                 │
   │                 └── Ramsey\Uuid::uuid4()->toString()
   │                         │
   │                         └── Returns random UUID v4
   │
   └── Testing → UuidGeneratorStub
                  │
                  ├── Has preset UUID? → Return it (one-time use)
                  │
                  └── No preset? → Generate random UUID v4
```

### Stub Behavior

| State | `generate()` Returns | After Call |
|-------|---------------------|------------|
| Preset UUID set | The preset value | Clears preset (null) |
| No preset | Random UUID v4 | Stays null |

The one-time-use pattern ensures each `setNextUuid()` call affects exactly one `generate()` call, making tests predictable.

## Architecture

```
UuidGeneratorInterface (contract)
├── UuidGenerator       - Production implementation (final)
└── UuidGeneratorStub   - Test double with preset capability
        │
        └── UuidContext - Behat step definitions
```

See `src/` for implementation details.

## Behat Integration

The library includes a Behat context for BDD testing:

```gherkin
Given next UUID is going to be "550e8400-e29b-41d4-a716-446655440000"
When I create a new order
Then the order ID should be "550e8400-e29b-41d4-a716-446655440000"
```

## Installation

```bash
composer require avodel/uuid
```

## External Dependencies

| Package | Purpose |
|---------|---------|
| `ramsey/uuid` | UUID v4 generation |
