<?php

namespace Avodel\WebBot\Extension\Captcha;

use Avodel\WebBot\Context\Context;
use Avodel\WebBot\Worker\ActionInterface;
use Avodel\WebDriver\WebDriver;
use Psr\Log\LoggerInterface;

final class SetActualUserAgentAction implements ActionInterface
{
    private bool $userAgentSet = false;

    public function __construct(
        private readonly WebDriver $webDriver,
        private readonly LoggerInterface $logger,
    )
    {
    }

    public function perform(WebDriver $webDriver, Context $context): void
    {
        $latestUserAgent = $this->getActualUserAgent();

        if ($latestUserAgent === null) {
            $this->logger->warning('Unable to get latest user agent.');
            return;
        }

        $this->webDriver->getDevToolsDriver()->setUserAgentOverride($latestUserAgent);
        $this->userAgentSet = true;
    }

    public function isApplicable(WebDriver $webDriver, Context $context): bool
    {
        if (!$context->getOptions()->isRequireActualUserAgent()) {
            return false;
        }

        if ($this->userAgentSet) {
            return false;
        }

        $latestUserAgent = $this->getActualUserAgent();

        if ($latestUserAgent === null) {
            return false;
        }

        $currentUserAgent = $webDriver->evaluateScript('navigator.userAgent');

        return $latestUserAgent !== $currentUserAgent;
    }

    /**
     * @return string|null
     */
    public function getActualUserAgent(): ?string
    {
        $userAgent = file_get_contents('https://capmonster.cloud/api/useragent/actual');

        if ($userAgent === false) {
            return null;
        }

        return $userAgent;
    }
}
