<?php

declare(strict_types=1);

namespace Avodel\WebBot\Command;

use Avodel\WebBot\Context\Options;
use Avodel\WebBot\Context\Schedule;
use Avodel\WebBot\Worker\Worker;
use Avodel\WebBot\Worker\WorkersRegistry;
use JsonException;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Command\SignalableCommandInterface;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use const FILTER_VALIDATE_BOOL;

#[AsCommand(name: 'webbot:run')]
class RunWebBotCommand extends Command implements SignalableCommandInterface
{
    private ?Worker $worker = null;

    public function __construct(
        private readonly WorkersRegistry $workerRegistry,
    )
    {
        parent::__construct();
    }

    protected function configure(): void
    {
        $this
            ->addArgument('profile', InputArgument::REQUIRED, 'Worker profile')
            ->addOption('time-limit', null, InputOption::VALUE_REQUIRED, 'Worker time limit in seconds. Infinite by default.')
            ->addOption('pause-between-user-actions', null, InputOption::VALUE_REQUIRED, 'Pause between user actions in seconds. Example: (1-6)')
            ->addOption('upstream-proxy-servers', null, InputOption::VALUE_REQUIRED, 'A list of proxy servers to upstream the traffic through.')
            ->addOption('upstream-proxy-shuffle', null, InputOption::VALUE_NONE, 'Shuffle upstream proxy servers')
            ->addOption('upstream-proxy-rotate-interval', null, InputOption::VALUE_REQUIRED, 'Rotate upstream proxy servers every X seconds.', null)
            ->addOption('actual-user-agent', null, InputOption::VALUE_NONE, 'Get and set the latest user agent.')
            ->addOption('tag', null, InputOption::VALUE_REQUIRED, 'Worker tag for logging.', null)
            ->addOption('schedule', null, InputOption::VALUE_REQUIRED, 'Working schedule.', 'mon-sun|00:00:00-23:59:59');
    }


    /**
     * @throws JsonException
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $optionsBuilder = Options::builder();

        if ($input->getOption('time-limit')) {
            $optionsBuilder->timeLimitSec((int)$input->getOption('time-limit'));
        }

        if ($input->getOption('pause-between-user-actions')) {
            $pauseBetweenUserActions = explode('-', $input->getOption('pause-between-user-actions'));
            $optionsBuilder->pauseBetweenUserActionsMinSec((int)$pauseBetweenUserActions[0]);
            $optionsBuilder->pauseBetweenUserActionsMaxSec((int)$pauseBetweenUserActions[1]);
        }

        if ($input->getOption('upstream-proxy-servers')) {
            $proxyServers = json_decode(file_get_contents($input->getOption('upstream-proxy-servers')), true, 512, JSON_THROW_ON_ERROR);

            if ($input->getOption('upstream-proxy-shuffle')) {
                $proxyServers = $this->shuffleArray($proxyServers);
            }

            $optionsBuilder->upstreamProxyServers($proxyServers);
        }

        if ($input->getOption('tag')) {
            $optionsBuilder->tag($input->getOption('tag'));
        }

        if ($input->getOption('upstream-proxy-rotate-interval')) {
            $optionsBuilder->rotateUpstreamProxyServerIntervalSec((int) $input->getOption('upstream-proxy-rotate-interval'));
        }

        $options = $optionsBuilder
            ->schedule(new Schedule($input->getOption('schedule')))
            ->actualUserAgent(filter_var($input->getOption('actual-user-agent'), FILTER_VALIDATE_BOOL))
            ->build();

        $this->worker = $this->workerRegistry->getWorker($input->getArgument('profile'));
        $this->worker->run($options);

        return Command::SUCCESS;
    }

    private function shuffleArray(array $array): array
    {
        $keys = array_keys($array);

        shuffle($keys);

        $shuffled = [];

        foreach ($keys as $key) {
            $shuffled[$key] = $array[$key];
        }

        return array_values($shuffled);
    }

    public function getSubscribedSignals(): array
    {
        return [SIGINT, SIGTERM];
    }

    public function handleSignal(int $signal, int|false $previousExitCode = 0): int|false
    {
        if (!$this->worker) {
            return 0;
        }

        $this->worker->stop();

        return false;
    }
}
