<?php

declare(strict_types=1);

namespace Avodel\WebBot\Extension\Pause;

use Avodel\WebBot\Context\Context;
use Avodel\WebBot\Worker\ActionInterface;
use Avodel\WebDriver\WebDriver;
use Override;
use Psr\Log\LoggerInterface;

final readonly class PauseBetweenUserActionsAction implements ActionInterface
{
    public function __construct(
        private LoggerInterface $logger,
    )
    {
    }

    #[Override]
    public function perform(WebDriver $webDriver, Context $context): void
    {
        $options = $context->getOptions();
        $lastUserActionPerformedSecondsAgo = $webDriver->getTimeSinceLastUserActionSec();

        $minSleep = max($options->getPauseBetweenUserActionsMinSec() - $lastUserActionPerformedSecondsAgo, 0);
        $maxSleep = max($options->getPauseBetweenUserActionsMaxSec() - $lastUserActionPerformedSecondsAgo, 0);

        if ($maxSleep <= 0) {
            return;
        }

        $sleepSeconds = random_int($minSleep, $maxSleep);
        $this->logger->debug('Pause between user actions.', ['seconds' => $sleepSeconds]);

        sleep($sleepSeconds);
    }

    #[Override]
    public function isApplicable(WebDriver $webDriver, Context $context): bool
    {
        if ($context->getOptions()->getPauseBetweenUserActionsMaxSec() <= 0) {
            return false;
        }

        $timeSinceLastUserAction = $webDriver->getTimeSinceLastUserActionSec();

        if ($timeSinceLastUserAction === null) {
            return false;
        }

        return $timeSinceLastUserAction < $context->getOptions()->getPauseBetweenUserActionsMaxSec();
    }
}
