<?php

namespace Avodel\WebBot\Context;

use DateTimeImmutable;

final readonly class Schedule
{
    private array $schedule;

    public function __construct(string $schedule)
    {
        $this->schedule = $this->parseSchedule($schedule);
    }

    private function parseSchedule(string $schedule): array
    {
        $scheduleItems = explode(',', $schedule);
        $parsedSchedule = [];

        foreach ($scheduleItems as $item) {
            [$days, $hours] = explode('|', $item);
            $dayRange = $this->parseDays($days);
            [$startTime, $endTime] = explode('-', $hours);

            foreach ($dayRange as $day) {
                $parsedSchedule[$day] = [
                    'start' => $startTime,
                    'end' => $endTime,
                ];
            }
        }

        return $parsedSchedule;
    }

    private function parseDays(string $days): array
    {
        $daysMap = [
            'mon' => 'mon',
            'tue' => 'tue',
            'wed' => 'wed',
            'thu' => 'thu',
            'fri' => 'fri',
            'sat' => 'sat',
            'sun' => 'sun',
        ];

        $result = [];
        if (str_contains($days, '-')) {
            [$startDay, $endDay] = explode('-', $days);
            $foundStart = false;

            foreach (array_keys($daysMap) as $day) {
                if ($day === $startDay) {
                    $foundStart = true;
                }
                if ($foundStart) {
                    $result[] = $day;
                }
                if ($day === $endDay) {
                    break;
                }
            }
        } else {
            $result[] = $daysMap[$days];
        }

        return $result;
    }

    public function isNowWithinSchedule(DateTimeImmutable $now): bool
    {
        $currentDay = strtolower($now->format('D'));
        $currentTime = $now->format('H:i:s');

        if (isset($this->schedule[$currentDay])) {
            $start = $this->schedule[$currentDay]['start'];
            $end = $this->schedule[$currentDay]['end'];

            return $currentTime >= $start && $currentTime <= $end;
        }

        return false;
    }
}
