<?php

namespace Avodel\WebBot\Extension\TimeLimit;

use Avodel\WebBot\Context\Context;
use Avodel\WebBot\Worker\ActionInterface;
use Avodel\WebDriver\WebDriver;
use DateTimeImmutable;
use Psr\Clock\ClockInterface;
use Psr\Log\LoggerInterface;

/**
 * Stops worker when the configured running time has passed.
 */
final class StopWorkerOnTimeLimitAction implements ActionInterface
{
    private ?DateTimeImmutable $startedAt = null;

    public function __construct(
        private readonly ClockInterface $clock,
        private readonly LoggerInterface $logger,
        private readonly ?StopWorkerHandlerInterface $stopWorkerHandler = null,
    )
    {
    }

    public function perform(WebDriver $webDriver, Context $context): void
    {
        if (!$this->startedAt) {
            $this->startedAt = $this->clock->now();

            return;
        }

        $context->getWorker()->stop();
        $this->logger->info('Worker stopped due to running time limit.');

        $this->stopWorkerHandler?->handle($context);
    }

    public function isApplicable(WebDriver $webDriver, Context $context): bool
    {
        if (!$this->startedAt) {
            return true;
        }

        $timeLimit = $context->getOptions()->getTimeLimitSec();

        if (!$timeLimit || $timeLimit <= 0) {
            return false;
        }

        return $this->clock->now() > $this->startedAt->modify('+' . $timeLimit . ' seconds');
    }
}
