<?php

declare(strict_types=1);

namespace Avodel\WebBot\Worker;

use Avodel\ProxyHandler\MitmProxyHandler;
use Avodel\WebBot\Context\Context;
use Avodel\WebBot\Context\ContextHolder;
use Avodel\WebBot\Context\Options;
use Avodel\WebBot\Proxy\ProxyManager;
use Avodel\WebDriver\WebDriver;
use Psr\Clock\ClockInterface;
use Psr\Log\LoggerInterface;

final class Worker
{
    private bool $shouldStop = false;

    public function __construct(
        /**
         * @var array<ActionInterface>
         */
        private readonly array $actions,
        /**
         * @var array<ExceptionHandlerInterface>
         */
        private readonly array $exceptionHandlers,
        private readonly LoggerInterface $logger,
        private readonly WebDriver $webDriver,
        private readonly ContextHolder $contextHolder,
        private readonly ClockInterface $clock,
    )
    {
    }

    public function run(Options $options): void
    {
        if ($options->getUpstreamProxyServers() && $this->webDriver->getHttpProxy()) {
            $proxyManager = new ProxyManager(
                new MitmProxyHandler($this->webDriver->getHttpProxy()),
                $options->getUpstreamProxyServers(),
                $this->clock,
                $this->logger,
            );
        }

        $context = new Context($this, $proxyManager ?? null, $options);
        $this->contextHolder->setContext($context);

        while (!$this->shouldStop) {
            foreach ($this->actions as $action) {
                if ($this->shouldStop) {
                    break 2;
                }

                try {
                    if ($action->isApplicable($this->webDriver, $context)) {
                        $action->perform($this->webDriver, $context);

                        $this->logger->debug('Processed action.', ['action' => $action::class]);

                        continue 2;
                    }
                } catch (\Throwable $exception) {
                    $this->logger->debug('Action failed with exception.', ['action' => $action::class, 'exception' => $exception]);

                    foreach ($this->exceptionHandlers as $exceptionHandler) {
                        $handledState = $exceptionHandler->handleException($this->webDriver, $context, $exception);

                        if ($handledState) {
                            $this->logger->debug('Exception is handled.', ['exceptionHandler' => $exceptionHandler::class, 'exception' => $exception]);

                            continue 3;
                        }
                    }

                    $this->shouldStop = true;
                    $this->logger->warning('Finished worker due to unhandled exception.', ['exception' => $exception]);
                }
            }
        }

        if ($this->webDriver->isStarted()) {
            $this->webDriver->stop();
        }

        $this->logger->info('Finished run.');
    }

    public function stop(): void
    {
        $this->shouldStop = true;
    }
}
