<?php

declare(strict_types=1);

namespace Avodel\WebBot\Command;

use Avodel\WebBot\Context\Options;
use Avodel\WebBot\Context\Schedule;
use Avodel\WebBot\Worker\Worker;
use Avodel\WebBot\Worker\WorkersRegistry;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Command\SignalableCommandInterface;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use const FILTER_VALIDATE_BOOL;

#[AsCommand(name: 'webbot:run')]
class RunWebBotCommand extends Command implements SignalableCommandInterface
{
    private ?Worker $worker = null;

    public function __construct(
        private readonly WorkersRegistry $workerRegistry,
    )
    {
        parent::__construct();
    }

    protected function configure(): void
    {
        $this
            ->addArgument('profile', InputArgument::REQUIRED, 'Worker profile')
            ->addOption('time-limit', null, InputOption::VALUE_REQUIRED, 'Worker time limit in seconds. Infinite by default.')
            ->addOption('pause-between-user-actions', null, InputOption::VALUE_REQUIRED, 'Pause between user actions in ms. Example: (1000-6000)')
            ->addOption('tag', null, InputOption::VALUE_REQUIRED, 'Worker tag for logging.', null)
            ->addOption('schedule', null, InputOption::VALUE_REQUIRED, 'Working schedule.', 'mon-sun|00:00:00-23:59:59');
    }


    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $optionsBuilder = Options::builder();

        if ($input->getOption('time-limit')) {
            $optionsBuilder->timeLimitSec((int)$input->getOption('time-limit'));
        }

        if ($input->getOption('pause-between-user-actions')) {
            $pauseBetweenUserActions = explode('-', $input->getOption('pause-between-user-actions'));
            $optionsBuilder->pauseBetweenUserActionsMinMs((int)$pauseBetweenUserActions[0]);
            $optionsBuilder->pauseBetweenUserActionsMaxMs((int)$pauseBetweenUserActions[1]);
        }

        if ($input->getOption('tag')) {
            $optionsBuilder->tag($input->getOption('tag'));
        }

        $options = $optionsBuilder
            ->schedule(new Schedule($input->getOption('schedule')))
            ->build();

        $this->worker = $this->workerRegistry->getWorker($input->getArgument('profile'));
        $this->worker->run($options);

        return Command::SUCCESS;
    }

    public function getSubscribedSignals(): array
    {
        return [SIGINT, SIGTERM];
    }

    public function handleSignal(int $signal, int|false $previousExitCode = 0): int|false
    {
        if (!$this->worker) {
            return 0;
        }

        $this->worker->stop();

        return false;
    }
}
