<?php

declare(strict_types=1);

namespace Avodel\WebBot\Command;

use Avodel\WebBot\Context\Options;
use Avodel\WebBot\Worker\Worker;
use Avodel\WebBot\Worker\WorkersRegistry;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Command\SignalableCommandInterface;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

#[AsCommand(name: 'webbot:run')]
class RunWebBotCommand extends Command implements SignalableCommandInterface
{
    private ?Worker $worker = null;

    public function __construct(
        private readonly WorkersRegistry $workerRegistry,
    )
    {
        parent::__construct();
    }

    protected function configure(): void
    {
        $this
            ->addArgument('profile', InputArgument::REQUIRED, 'Worker profile')
            ->addOption('ttl', null, InputOption::VALUE_REQUIRED, 'Worker time limit in seconds. Infinite by default.')
            ->addOption('pause-between-user-actions', null, InputOption::VALUE_REQUIRED, 'Pause between user actions in ms. Example: (1000-6000)')
        ;
    }


    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $optionsBuilder = Options::builder();

        if ($input->getOption('ttl')) {
            $optionsBuilder->ttl((int)$input->getOption('ttl'));
        }

        if ($input->getOption('pause-between-user-actions')) {
            $pauseBetweenUserActions = explode('-', $input->getOption('pause-between-user-actions'));
            $optionsBuilder->pauseBetweenUserActionsMinMs((int)$pauseBetweenUserActions[0]);
            $optionsBuilder->pauseBetweenUserActionsMaxMs((int)$pauseBetweenUserActions[1]);
        }

        $options = $optionsBuilder->build();

        $this->worker = $this->workerRegistry->getWorker($input->getArgument('profile'));
        $this->worker->run($options);

        return Command::SUCCESS;
    }

    public function getSubscribedSignals(): array
    {
        return [SIGINT, SIGTERM];
    }

    public function handleSignal(int $signal, int|false $previousExitCode = 0): int|false
    {
        if (!$this->worker) {
            return 0;
        }

        $this->worker->stop();

        return false;
    }
}
