<?php

declare(strict_types=1);

namespace Avodel\WebBot\Extension\WebDriver;

use Avodel\WebBot\Context\Context;
use Avodel\WebBot\Worker\ExceptionHandlerInterface;
use Avodel\WebDriver\Driver\WebDriver;
use Override;
use Psr\Log\LoggerInterface;
use Throwable;

final readonly class FallbackExceptionHandler implements ExceptionHandlerInterface
{
    public function __construct(
        private LoggerInterface $logger,
        private int $backoffDelayMs,
    )
    {
    }

    #[Override]
    public function handleException(WebDriver $webDriver, Context $context, Throwable $exception): bool
    {
        $actions = $context->getPerformedActions();
        $previousAction = $context->getPerformedActions()[count($actions) - 2] ?? null;

        if ($exception instanceof UnexpectedContextException && !$previousAction instanceof FallbackAction) {
            $this->logger->debug('Unexpected context.');

            usleep($this->backoffDelayMs * 1000);

            return true;
        }

        $this->logger->error('Unhandled exception.', [
            'exception' => $exception,
            'debug' => $webDriver->getDebugger()->dump(true),
        ]);

        usleep($this->backoffDelayMs * 1000);

        $webDriver->stop();
        $context->clearStates();

        return true;
    }
}
