<?php

namespace Avodel\WebBot\Extension\Ajax;

use Avodel\WebBot\Context\Context;
use Avodel\WebBot\Worker\ActionInterface;
use Avodel\WebDriver\Components\Ajax\Exception\AjaxResponseWasNotReceivedException;
use Avodel\WebDriver\Components\Frames\SwitchToFrameException;
use Avodel\WebDriver\Driver\WebDriver;
use Psr\Log\LoggerInterface;

/**
 * Waits until all AJAX requests are finished in all frames.
 */
final readonly class WaitUntilAjaxRequestsFinishedAction implements ActionInterface
{
    public function __construct(
        private LoggerInterface $logger,
    )
    {
    }

    /**
     * @throws AjaxResponseWasNotReceivedException
     */
    public function perform(WebDriver $webDriver, Context $context): void
    {
        $framesHandler = $webDriver->getFramesHandler();
        $frames = $framesHandler->getFrames();
        $ajaxHandler = $webDriver->getAjaxHandler();

        if ($ajaxHandler->isAjaxRequestActive()) {
            $this->logger->info('Waiting for AJAX requests to be finished.');
            $ajaxHandler->waitUntilAllAjaxRequestsAreFinished();

            return;
        }

        foreach ($frames as $frame) {
            try {
                $framesHandler->switchToIFrame($frame->getPath());

                // @phpstan-ignore-next-line
                if ($ajaxHandler->isAjaxRequestActive()) {
                    $this->logger->info('Waiting for frame AJAX requests to be finished.', ['frame' => $frame->getPath()]);
                    $ajaxHandler->waitUntilAllAjaxRequestsAreFinished();
                    return;
                }
            } catch (SwitchToFrameException) {
                continue;
            } finally {
                $framesHandler->switchToMainWindow();
            }
        }
    }

    public function isApplicable(WebDriver $webDriver, Context $context): bool
    {
        $framesHandler = $webDriver->getFramesHandler();
        $frames = $framesHandler->getFrames();
        $ajaxHandler = $webDriver->getAjaxHandler();

        if ($ajaxHandler->isAjaxRequestActive()) {
            return true;
        }

        foreach ($frames as $frame) {
            try {
                $framesHandler->switchToIFrame($frame->getPath());

                // @phpstan-ignore-next-line
                if ($ajaxHandler->isAjaxRequestActive()) {
                    return true;
                }
            } catch (SwitchToFrameException) {
                continue;
            } finally {
                $framesHandler->switchToMainWindow();
            }
        }

        return false;
    }
}
