<?php

namespace Avodel\WebBot\Extension\Captcha;

use Avodel\WebBot\Context\Context;
use Avodel\WebBot\Worker\ActionInterface;
use Avodel\WebDriver\Components\CaptchaVerifier\Exception\CaptchaVerificationFailedException;
use Avodel\WebDriver\Components\Frames\SwitchToFrameException;
use Avodel\WebDriver\Driver\WebDriver;

final readonly class WaitForCaptchaWidgetReadinessAction implements ActionInterface
{
    private const int MAX_WAITING_TIME_MS = 30_000;

    /**
     * @throws CaptchaVerificationFailedException
     * @throws SwitchToFrameException
     */
    public function perform(WebDriver $webDriver, Context $context): void
    {
        $frames = $webDriver->getFramesHandler()->getFrames();
        $recaptchaV2Anchor = false;
        $recaptchaV2BFrame = false;

        foreach ($frames as $frame) {
            if (str_starts_with($frame->getSrc(), 'https://www.google.com/recaptcha/api2/anchor') && str_contains($frame->getSrc(), 'size=normal')) {
                $recaptchaV2Anchor = true;

                try {
                    $webDriver->getFramesHandler()->switchToIFrame($frame->getPath());
                    $waited = $webDriver->wait(self::MAX_WAITING_TIME_MS, 'document.querySelector(\'#rc-anchor-container\') !== null');

                    if (!$waited) {
                        throw new CaptchaVerificationFailedException('RecaptchaV2 checkbox was not rendered.');
                    }
                } finally {
                    $webDriver->getFramesHandler()->switchToMainWindow();
                }
            }

            if (str_starts_with($frame->getSrc(), 'https://www.google.com/recaptcha/api2/bframe')) {
                $recaptchaV2BFrame = true;
            }

            if (str_starts_with($frame->getSrc(), 'https://newassets.hcaptcha.com/captcha/') && str_contains($frame->getSrc(), '#frame=checkbox')) {
                try {
                    $webDriver->getFramesHandler()->switchToIFrame($frame->getPath());
                    $waited = $webDriver->wait(self::MAX_WAITING_TIME_MS, 'document.querySelector(\'#anchor\') !== null');

                    if (!$waited) {
                        throw new CaptchaVerificationFailedException('HCaptcha checkbox was not rendered.');
                    }
                } finally {
                    $webDriver->getFramesHandler()->switchToMainWindow();
                }
            }
        }

        if ($recaptchaV2Anchor && !$recaptchaV2BFrame) {
            $start = microtime(true);
            $frames = $webDriver->getFramesHandler()->getFrames();

            do {
                foreach ($frames as $frame) {
                    if (str_starts_with($frame->getSrc(), 'https://www.google.com/recaptcha/api2/bframe')) {
                        return;
                    }
                }

                $frames = $webDriver->getFramesHandler()->getFrames();
                sleep(1);
            } while (microtime(true) - $start < self::MAX_WAITING_TIME_MS / 1000);

            throw new CaptchaVerificationFailedException('reCAPTCHA V2 bFrame was not loaded.');
        }
    }

    public function isApplicable(WebDriver $webDriver, Context $context): bool
    {
        $frames = $webDriver->getFramesHandler()->getFrames();
        $recaptchaV2Anchor = false;
        $recaptchaV2BFrame = false;

        foreach ($frames as $frame) {
            if (str_starts_with($frame->getSrc(), 'https://www.google.com/recaptcha/api2/anchor') && str_contains($frame->getSrc(), 'size=normal')) {
                $recaptchaV2Anchor = true;

                try {
                    $webDriver->getFramesHandler()->switchToIFrame($frame->getPath());
                    $checkbox = $webDriver->getPage()->find('css', '#rc-anchor-container');

                    if (!$checkbox) {
                        return true;
                    }
                } finally {
                    $webDriver->getFramesHandler()->switchToMainWindow();
                }
            }

            if (str_starts_with($frame->getSrc(), 'https://www.google.com/recaptcha/api2/bframe')) {
                $recaptchaV2BFrame = true;
            }

            if (str_starts_with($frame->getSrc(), 'https://newassets.hcaptcha.com/captcha/') && str_contains($frame->getSrc(), '#frame=checkbox')) {
                try {
                    $webDriver->getFramesHandler()->switchToIFrame($frame->getPath());
                    $checkbox = $webDriver->getPage()->find('css', '#anchor');

                    if (!$checkbox) {
                        return true;
                    }
                } finally {
                    $webDriver->getFramesHandler()->switchToMainWindow();
                }
            }
        }

        return $recaptchaV2Anchor && !$recaptchaV2BFrame;
    }
}
