<?php

declare(strict_types=1);

namespace Avodel\WebBot\Worker;

use Avodel\WebBot\Context\Context;
use Avodel\WebBot\Context\ContextHolder;
use Avodel\WebBot\Context\ContextImpl;
use Avodel\WebBot\Context\Options;
use Avodel\WebBot\Extension\TimeLimit\StopWorkerHandlerInterface;
use Avodel\WebDriver\Driver\WebDriver;
use Avodel\WebDriver\Driver\WebDriverFactory;
use Psr\Log\LoggerInterface;

final class Worker
{
    private bool $shouldStop = false;

    public function __construct(
        /**
         * @var array<ActionInterface>
         */
        private readonly array $actions,
        /**
         * @var array<ExceptionHandlerInterface>
         */
        private readonly array $exceptionHandlers,
        private readonly LoggerInterface $logger,
        private readonly WebDriverFactory $webDriverFactory,
        private readonly ContextHolder $contextHolder,
        private readonly ?StopWorkerHandlerInterface $stopWorkerHandler = null,
    )
    {
    }

    public function run(Options $options): void
    {
        $webDriver = $this->webDriverFactory->create();
        $context = new ContextImpl($this, $options);
        $this->contextHolder->setContext($context);

        try {
            $this->doRun($webDriver, $context);
        } finally {
            $this->stopWorkerHandler?->handle($context);

            if ($webDriver->isStarted()) {
                $webDriver->stop();
            }
        }

        $this->logger->info('Finished run.');
    }

    private function doRun(WebDriver $webDriver, ContextImpl $context): void
    {
        while (!$this->shouldStop) {
            foreach ($this->actions as $action) {
                if ($this->shouldStop) {
                    break 2;
                }

                try {
                    if ($action->isApplicable($webDriver, $context)) {
                        try {
                            $action->perform($webDriver, $context);
                        } finally {
                            $context->addPerformedAction($action);
                        }

                        $this->logger->debug('Performed action.', ['action' => $action::class]);

                        continue 2;
                    }
                } catch (\Throwable $exception) {
                    $this->logger->debug('Action failed with exception.', ['action' => $action::class, 'exception' => $exception]);

                    foreach ($this->exceptionHandlers as $exceptionHandler) {
                        $handledState = $exceptionHandler->handleException($webDriver, $context, $exception);

                        if ($handledState) {
                            $this->logger->debug('Exception is handled.', ['exceptionHandler' => $exceptionHandler::class, 'exception' => $exception]);

                            continue 3;
                        }
                    }

                    $this->shouldStop = true;
                    $this->logger->warning('Finished worker due to unhandled exception.', ['exception' => $exception]);
                }
            }
        }

        $this->logger->info('Finished run.');
    }

    public function stop(): void
    {
        $this->shouldStop = true;
    }
}
