<?php

declare(strict_types=1);

namespace Avodel\WebBot\Extension\Pause;

use Avodel\WebBot\Context\Context;
use Avodel\WebBot\Worker\ActionInterface;
use Avodel\WebDriver\Driver\WebDriver;
use Override;
use Psr\Log\LoggerInterface;

final readonly class PauseBetweenUserActionsAction implements ActionInterface
{
    public function __construct(
        private LoggerInterface $logger,
        private PauseMouseActionsSimulator $pauseMouseActionsSimulator
    )
    {
    }

    #[Override]
    public function perform(WebDriver $webDriver, Context $context): void
    {
        $options = $context->getOptions();
        $lastUserActionPerformedSecondsAgo = $webDriver->getTimeSinceLastUserActionMs();

        $minSleep = max($options->getPauseBetweenUserActionsMinMs() - $lastUserActionPerformedSecondsAgo, 0);
        $maxSleep = max($options->getPauseBetweenUserActionsMaxMs() - $lastUserActionPerformedSecondsAgo, 0);

        if ($maxSleep <= 0) {
            return;
        }

        $sleepMs = random_int($minSleep, $maxSleep);

        $this->logger->debug('Pause between user actions.', ['ms' => $sleepMs]);
        try {
            $this->pauseMouseActionsSimulator->simulateNaturalBehaviour($webDriver, $sleepMs);
        } catch (\Exception $e) {
            $this->logger->warning('Failed to simulate natural mouse behaviour while pausing.', ['exc' => $e]);
        }
    }

    #[Override]
    public function isApplicable(WebDriver $webDriver, Context $context): bool
    {
        if ($context->getOptions()->getPauseBetweenUserActionsMaxMs() <= 0) {
            return false;
        }

        $timeSinceLastUserAction = $webDriver->getTimeSinceLastUserActionMs();

        if ($timeSinceLastUserAction === null) {
            return false;
        }

        return $timeSinceLastUserAction < $context->getOptions()->getPauseBetweenUserActionsMinMs();
    }
}
