<?php

namespace Avodel\WebBot\Extension\Pause;

use Avodel\WebDriver\Driver\WebDriver;
use Avodel\WebDriver\Mouse\Coordinate;

class PauseMouseActionsSimulator
{
    private const string ACTION_WAIT_LITTLE = 'WAIT_LITTLE';
    private const string ACTION_WAIT_MEDIUM = 'WAIT_MEDIUM';
    private const string ACTION_WAIT_LONG = 'WAIT_LONG';
    private const string ACTION_SCROLL_LITTLE = 'SCROLL_LITTLE';
    private const string ACTION_SCROLL_MEDIUM = 'SCROLL_MEDIUM';
    private const string ACTION_SCROLL_TO_TOP = 'SCROLL_TO_TOP';
    private const string ACTION_MOUSE_MOVE_LITTLE = 'MOUSE_MOVE_LITTLE';
    private const string ACTION_MOUSE_MOVE_MEDIUM = 'MOUSE_MOVE_MEDIUM';
    private const string ACTION_SCROLL_LARGE = 'SCROLL_LARGE';

    private const array ACTION_THRESHOLDS = [
        self::ACTION_WAIT_LITTLE => [
            'minMs' => 500,
            'maxMs' => 1500,
        ],
        self::ACTION_SCROLL_LITTLE => [
            'minPx' => 100,
            'maxPx' => 250,
        ],
        self::ACTION_MOUSE_MOVE_LITTLE => [
            'minPx' => 100,
            'maxPx' => 250,
        ],
        self::ACTION_WAIT_MEDIUM => [
            'minMs' => 1500,
            'maxMs' => 4500,
        ],
        self::ACTION_SCROLL_MEDIUM => [
            'minPx' => 250,
            'maxPx' => 500,
        ],
        self::ACTION_MOUSE_MOVE_MEDIUM => [
            'minPx' => 250,
            'maxPx' => 500,
        ],
        self::ACTION_WAIT_LONG => [
            'minMs' => 4500,
            'maxMs' => 10000,
        ],
        self::ACTION_SCROLL_LARGE => [
            'minPx' => 500,
            'maxPx' => 1000,
        ],
    ];

    private const array WEIGHT_TABLES =
        [
            [
                "tableId" => 1,
                "weight" => 10,
                "action" => self::ACTION_WAIT_LITTLE,
            ],
            [
                "tableId" => 1,
                "weight" => 2,
                "action" => self::ACTION_SCROLL_LITTLE,
            ],
            [
                "tableId" => 1,
                "weight" => 2,
                "action" => self::ACTION_MOUSE_MOVE_LITTLE,
            ],
            [
                "tableId" => 2,
                "weight" => 10,
                "action" => self::ACTION_WAIT_LITTLE
            ],
            [
                "tableId" => 2,
                "weight" => 10,
                "action" => self::ACTION_WAIT_MEDIUM
            ],
            [
                "tableId" => 2,
                "weight" => 8,
                "action" => self::ACTION_WAIT_LONG
            ],
            [
                "tableId" => 2,
                "weight" => 5,
                "action" => self::ACTION_SCROLL_LITTLE
            ],
            [
                "tableId" => 2,
                "weight" => 5,
                "action" => self::ACTION_SCROLL_MEDIUM
            ],
            [
                "tableId" => 2,
                "weight" => 5,
                "action" => self::ACTION_SCROLL_TO_TOP
            ],
            [
                "tableId" => 2,
                "weight" => 5,
                "action" => self::ACTION_MOUSE_MOVE_MEDIUM
            ],
            [
                "tableId" => 2,
                "weight" => 5,
                "action" => self::ACTION_MOUSE_MOVE_LITTLE
            ],
            [
                "tableId" => 3,
                "weight" => 10,
                "action" => self::ACTION_WAIT_LITTLE
            ],
            [
                "tableId" => 3,
                "weight" => 10,
                "action" => self::ACTION_WAIT_MEDIUM
            ],
            [
                "tableId" => 3,
                "weight" => 10,
                "action" => self::ACTION_WAIT_LONG
            ],
            [
                "tableId" => 3,
                "weight" => 5,
                "action" => self::ACTION_SCROLL_LITTLE
            ],
            [
                "tableId" => 3,
                "weight" => 5,
                "action" => self::ACTION_SCROLL_MEDIUM
            ],
            [
                "tableId" => 3,
                "weight" => 2,
                "action" => self::ACTION_SCROLL_TO_TOP
            ],
            [
                "tableId" => 3,
                "weight" => 5,
                "action" => self::ACTION_MOUSE_MOVE_MEDIUM
            ],
            [
                "tableId" => 3,
                "weight" => 5,
                "action" => self::ACTION_MOUSE_MOVE_LITTLE
            ],
            [
                "tableId" => 3,
                "weight" => 1,
                "action" => self::ACTION_SCROLL_TO_TOP
            ]
        ];

    private const array PAUSE_THRESHOLDS = [
        [
            "minPauseThresholdMs" => 0,
            "maxPauseThresholdMs" => 2000,
            "weightTableId" => 1
        ],
        [
            "minPauseThresholdMs" => 2000,
            "maxPauseThresholdMs" => 4500,
            "weightTableId" => 2
        ],
        [
            "minPauseThresholdMs" => 4500,
            "maxPauseThresholdMs" => 1000000,
            "weightTableId" => 3
        ]
    ];

    /**
     * @throws \Exception
     */
    public function simulateNaturalBehaviour(WebDriver $webDriver, int $pauseWaitMs): void
    {
        $startedAt = (int) (microtime(true) * 1000);
        $pauseLeft = $pauseWaitMs;

        while ($pauseLeft > 0) {
            $threshold = $this->getThreshold($pauseLeft);
            $weightTable = $this->getRandomElementByWeightAndTableId($threshold['weightTableId']);
            $this->performAction($webDriver, $weightTable['action'], $pauseLeft);

            $now = (int) (microtime(true) * 1000);
            $activityTime = $now - $startedAt;
            $pauseLeft = max(0, $pauseWaitMs - $activityTime);
        }
    }

    private function performAction(WebDriver $webDriver,string $action, int $pauseLeft): void
    {
        switch ($action) {
            case self::ACTION_WAIT_LITTLE:
            case self::ACTION_WAIT_MEDIUM:
            case self::ACTION_WAIT_LONG:
                $sleep = random_int(self::ACTION_THRESHOLDS[$action]['minMs'], self::ACTION_THRESHOLDS[$action]['maxMs']);
                $actualSleepMs = min($sleep, $pauseLeft);
                usleep($actualSleepMs * 1000);
                break;
            case self::ACTION_SCROLL_LITTLE:
            case self::ACTION_SCROLL_MEDIUM:
            case self::ACTION_SCROLL_LARGE:
                $scrollLengthPx = random_int(self::ACTION_THRESHOLDS[$action]['minPx'], self::ACTION_THRESHOLDS[$action]['maxPx']);
                $this->performScroll($webDriver, $scrollLengthPx);
                break;
            case self::ACTION_MOUSE_MOVE_LITTLE:
            case self::ACTION_MOUSE_MOVE_MEDIUM:
                $mouseMoveLengthPx = random_int(self::ACTION_THRESHOLDS[$action]['minPx'], self::ACTION_THRESHOLDS[$action]['maxPx']);
                $this->performMouseMove($webDriver, $mouseMoveLengthPx);
                break;
            case self::ACTION_SCROLL_TO_TOP:
                $currentCoordinate = $webDriver->getMouse()->getScrollCoordinate();
                $webDriver->getMouse()->scroll(new Coordinate($currentCoordinate->getX(), 0));
                break;
        }
    }

    private function getRandomElementByWeightAndTableId(int $tableId)
    {
        $filteredElements = array_filter(self::WEIGHT_TABLES, function ($element) use ($tableId) {
            return $element['tableId'] === $tableId;
        });

        if (empty($filteredElements)) {
            throw new \Exception('No elements found for tableId ' . $tableId);
        }

        $totalWeight = array_sum(array_column($filteredElements, 'weight'));
        $random = random_int(1, $totalWeight);

        foreach ($filteredElements as $element) {
            if ($random <= $element['weight']) {
                return $element;
            }
            $random -= $element['weight'];
        }

        throw new \Exception('Random element not found');
    }


    private function getThreshold(int $pauseLeft): array
    {
        foreach (self::PAUSE_THRESHOLDS as $threshold) {
            if ($threshold['minPauseThresholdMs'] <= $pauseLeft && $threshold['maxPauseThresholdMs'] >= $pauseLeft) {
                return $threshold;
            }
        }

        throw new \Exception('Threshold not found');
    }

    private function performMouseMove(WebDriver $webDriver, int $mouseMoveLengthPx): void
    {
        $pageCoordinatesData = $this->getPageCoordinatesData($webDriver);

        $currentCoordinate = $webDriver->getMouse()->getMouseCoordinate();
        $currentX = $currentCoordinate->getX();
        $currentY = $currentCoordinate->getY();

        $windowHeight = $pageCoordinatesData['windowHeight'];
        $scrollTop = $pageCoordinatesData['scrollTop'];
        $windowWidth = $pageCoordinatesData['windowWidth'];

        $centerX = $windowWidth / 2;
        $centerY = $scrollTop + $windowHeight / 2;

        $directionX = $centerX > $currentX ? 1 : -1;
        $directionY = $centerY > $currentY ? 1 : -1;

        //Vertical scroll
        if (random_int(0, 10) < 6) {
            $targetY = $currentY + ($mouseMoveLengthPx * $directionY);
            $targetY = max($scrollTop, min($targetY, $scrollTop + $windowHeight - 1));
            $targetX = max(0, $currentX + random_int(-5, 5));
        } else {
            //Horizontal scroll
            $targetX = $currentX + ($mouseMoveLengthPx * $directionX);
            $targetX = max(0, min($targetX, $windowWidth - 1));
            $targetY = max(0, $currentY + random_int(-5, 5));
        }

        $webDriver->getMouse()->move(new Coordinate($targetX, $targetY));
    }

    private function performScroll(WebDriver $webDriver, int $scrollY): void
    {
        $pageCoordinatesData = $this->getPageCoordinatesData($webDriver);

        $scrollPositionY = $pageCoordinatesData['scrollTop'];
        $documentHeight = $pageCoordinatesData['scrollHeight'];
        $windowHeight = $pageCoordinatesData['windowHeight'];

        $canScrollDown = $scrollPositionY + $windowHeight < $documentHeight;
        $canScrollUp = $scrollPositionY > 0;

        $currentCoordinate = $webDriver->getMouse()->getScrollCoordinate();
        if ($canScrollUp && $canScrollDown) {
            $scrollY = random_int(0, 1) ? $scrollY : -$scrollY;
        } elseif ($canScrollDown) {
            $scrollY = $currentCoordinate->getY() + $scrollY;
        } elseif ($canScrollUp) {
            $scrollY = $currentCoordinate->getY() - $scrollY;
        }

        $webDriver->getMouse()->scroll(new Coordinate($currentCoordinate->getX(), $scrollY));
    }

    private function getPageCoordinatesData(WebDriver $webDriver): array
    {
        return $webDriver->evaluateScript('return {
            scrollTop: window.scrollY,
            scrollHeight: document.documentElement.scrollHeight,
            windowHeight: window.innerHeight,
            windowWidth: window.innerWidth
        };');
    }
}