<?php

namespace Avodel\WebDriver\Components\CaptchaVerifier\Verifier;

use Avodel\WebDriver\Components\CaptchaVerifier\Exception\CaptchaVerificationFailedException;
use Avodel\WebDriver\Components\CaptchaVerifier\Solver\HCaptchaSolverInterface;
use Avodel\WebDriver\Components\Ajax\AjaxUtil;
use Avodel\WebDriver\Components\Frames\Frame;
use Avodel\WebDriver\Components\Frames\FramesHelper;
use Behat\Mink\Session;
use Exception;
use Webmozart\Assert\Assert;

final readonly class HCaptchaCaptchaVerifier implements CaptchaVerifierInterface
{
    public function __construct(
        private FramesHelper $framesHelper,
        private HCaptchaSolverInterface $hCaptchaSolver,
        private AjaxUtil $ajaxUtil,
    )
    {
    }

    public function verify(Session $session): void
    {
        try {
            $this->doVerify($session);
        } catch (Exception $e) {
            throw new CaptchaVerificationFailedException('HCaptcha verification failed.', previous: $e);
        } finally {
            $this->framesHelper->switchToMainWindow($session);
        }
    }

    private function doVerify(Session $session): void
    {
        [$checkboxFrame, $challengeFrame] = $this->getHCaptchaFrames($session);

        Assert::notNull($checkboxFrame, 'Checkbox frame was not found.');
        Assert::notNull($challengeFrame, 'Challenge frame was not found.');

        $parsedUrl = parse_url($challengeFrame->getSrc());
        $query = $parsedUrl['fragment'] ?? '';
        parse_str($query, $params);

        $siteKey = $params['sitekey'] ?? null;
        Assert::notEmpty($siteKey, 'Site key was not found.');

        $userAgent = $session->evaluateScript('window.navigator.userAgent');

        $solution = $this->hCaptchaSolver->getSolution(
            $session->getCurrentUrl(),
            $siteKey,
            $userAgent
        );

        $this->framesHelper->switchToIFrame($session, $challengeFrame->getPath());
        $this->ajaxUtil->overrideAjaxResponse(
            $session,
            'POST',
            'https://api.hcaptcha.com/getcaptcha/' . $siteKey,
            '{"pass": true,"generated_pass_UUID": "' . $solution . '","expiration": 120}'
        );

        $this->framesHelper->switchToIFrame($session, $checkboxFrame->getPath());
        $session->executeScript('document.getElementById(\'anchor\').click()');

        $chekVerificationStatusScript = <<<JS
(function() {
    var elements = document.querySelectorAll('.check');
    
    // means the page might get reloaded and no checkbox was found
    if (elements.length === 0) {
        return true;
    }

    for (var i = 0; i < elements.length; i++) {
        if (getComputedStyle(elements[i]).display === 'block') {
            return true;
        }
    }
    return false;
})();
JS;
        $session->wait(2000, $chekVerificationStatusScript);
        Assert::true($session->evaluateScript($chekVerificationStatusScript), 'Checkbox was not checked.');

        $this->framesHelper->switchToMainWindow($session);
    }

    private function isCheckboxChecked(Session $session): bool
    {
        $checkboxCheckedScript = <<<JS
(function() {
    var elements = document.querySelectorAll('.check');
    
    for (var i = 0; i < elements.length; i++) {
        if (getComputedStyle(elements[i]).display === 'block') {
            return true;
        }
    }
    return false;
})();
JS;

        return $session->evaluateScript($checkboxCheckedScript);
    }

    public function isVerificationRequired(Session $session): bool
    {
        [$checkboxFrame, $challengeFrame] = $this->getHCaptchaFrames($session);

        if (!$challengeFrame || !$checkboxFrame) {
            return false;
        }

        try {
            $this->framesHelper->switchToIFrame($session, $checkboxFrame->getPath());

            return !$this->isCheckboxChecked($session);
        } finally {
            $this->framesHelper->switchToMainWindow($session);
        }
    }

    /**
     * @return array<Frame|null>
     */
    private function getHCaptchaFrames(Session $session): array
    {
        $allFrames = $this->framesHelper->getFrames($session);
        $checkboxFrame = null;
        $challengeFrame = null;

        foreach ($allFrames as $frame) {
            if (!str_starts_with($frame->getSrc(), 'https://newassets.hcaptcha.com/captcha/')) {
                continue;
            }

            if (strpos($frame->getSrc(), '#frame=checkbox') > 0) {
                $checkboxFrame = $frame;
                continue;
            }

            if (strpos($frame->getSrc(), '#frame=challenge') > 0) {
                $challengeFrame = $frame;
            }
        }

        return [$checkboxFrame, $challengeFrame];
    }
}
