<?php

namespace Avodel\WebDriver\Components\Frames;

use Behat\Mink\Session;

final readonly class FramesHelper
{
    /**
     * @return Frame[]
     */
    public function getFrames(Session $session): array
    {
        $session->switchToIFrame(null);

        try {
            $frames = $this->getFramesRecursive($session, []);
        } finally {
            $session->switchToIFrame(null);
        }

        return $frames;
    }

    private function getFramesRecursive(Session $session, array $path): array
    {
        $script = <<<JS
return Array.from(document.getElementsByTagName('iframe')).map((iframe, index) => {
    return { 
        index: index,
        id: iframe.id,
        name: iframe.name,
        src: iframe.src 
    };
})
JS;

        $framesArray = $session->getDriver()->evaluateScript($script);
        $response = [];

        foreach ($framesArray as $frame) {
            $currentPath = array_merge($path, [$frame['index']]);
            $response[] = new Frame($currentPath, $frame['id'], $frame['name'], $frame['src']);

            $session->switchToIFrame($frame['index']);
            $response = array_merge($response, $this->getFramesRecursive($session, $currentPath));

            $this->doSwitchToIFrame($session, $path);
        }

        return $response;
    }

    private function doSwitchToIFrame(Session $session, array $path): void
    {
        $session->switchToIFrame(null);

        foreach ($path as $p) {
            $session->switchToIFrame($p);
        }
    }

    public function switchToIFrame(Session $session, array $path): void
    {
        try {
            $this->doSwitchToIFrame($session, $path);
        } catch (\Exception $e) {
            $session->switchToIFrame(null);

            throw $e;
        }
    }

    public function switchToIFrameAndBack(Session $session, array $path, callable $callback): void
    {
        try {
            $this->doSwitchToIFrame($session, $path);
            $callback();
        } finally {
            $session->switchToIFrame(null);
        }
    }

    public function switchToMainWindow(Session $session): void
    {
        $session->switchToIFrame(null);
    }

    /**
     * Checks whether frame is interactable or not.
     *
     * Interactable frame means that:
     * - frame is visible
     * - frame is not overlapped by other elements
     */
    public function isFrameInteractable(Session $session, array $path): bool
    {
        $iFrameIndex = array_key_last($path);
        unset($path[$iFrameIndex]); // remove last element

        if (count($path) > 0) {
            $this->switchToIFrame($session, $path);
        }

        $js = <<<JS
(function(index) {
    const iframe = document.getElementsByTagName('iframe')[index]; 

    const rect = iframe.getBoundingClientRect();
    const windowHeight = (window.innerHeight || document.documentElement.clientHeight);
    const windowWidth = (window.innerWidth || document.documentElement.clientWidth);

    const isVisible = (
        rect.width > 0 &&
        rect.height > 0 &&
        rect.bottom >= 0 &&
        rect.right >= 0 &&
        rect.top <= windowHeight &&
        rect.left <= windowWidth
    );

    const style = window.getComputedStyle(iframe);
    const isCSSVisible = (
        style.display !== 'none' &&
        style.visibility !== 'hidden' &&
        style.pointerEvents !== 'none'
    );

    const isNotOverlapped = (function() {
        const midpointX = rect.left + rect.width / 2;
        const midpointY = rect.top + rect.height / 2;

        const elementAtMidpoint = document.elementFromPoint(midpointX, midpointY);

        return elementAtMidpoint === iframe || iframe.contains(elementAtMidpoint);
    })();

    return isVisible && isCSSVisible && isNotOverlapped;
})($iFrameIndex);
JS;

        try {
            return $session->evaluateScript($js);
        } finally {
            $this->switchToMainWindow($session);
        }
    }
}
