<?php

namespace Avodel\WebDriver\Components\CaptchaVerifier\Verifier;

use Avodel\WebDriver\Components\Ajax\AjaxUtil;
use Avodel\WebDriver\Components\CaptchaVerifier\Exception\CaptchaVerificationFailedException;
use Avodel\WebDriver\Components\CaptchaVerifier\Solver\HCaptchaSolverInterface;
use Avodel\WebDriver\Components\Frames\Frame;
use Avodel\WebDriver\Components\Frames\FramesHelper;
use Behat\Mink\Session;
use Exception;
use Webmozart\Assert\Assert;

final readonly class HCaptchaCaptchaVerifier implements CaptchaVerifierInterface
{
    public function __construct(
        private FramesHelper $framesHelper,
        private HCaptchaSolverInterface $hCaptchaSolver,
        private AjaxUtil $ajaxUtil,
    )
    {
    }

    public function verify(Session $session): void
    {
        try {
            $this->doVerify($session);
        } catch (Exception $e) {
            throw new CaptchaVerificationFailedException('HCaptcha verification failed.', previous: $e);
        } finally {
            $this->framesHelper->switchToMainWindow($session);
        }
    }

    private function doVerify(Session $session): void
    {
        [$checkboxFrame, $challengeFrame] = $this->getHCaptchaFrames($session);

        Assert::notNull($checkboxFrame, 'Checkbox frame was not found.');
        Assert::notNull($challengeFrame, 'Challenge frame was not found.');

        $siteKey = $this->parseQueryValue($challengeFrame->getSrc(), 'sitekey');
        Assert::notEmpty($siteKey, 'Site key was not found.');

        $userAgent = $session->evaluateScript('window.navigator.userAgent');

        $solution = $this->hCaptchaSolver->getSolution(
            $session->getCurrentUrl(),
            $siteKey,
            $userAgent
        );

        $this->framesHelper->switchToIFrame($session, $challengeFrame->getPath());
        $this->ajaxUtil->overrideAjaxResponse(
            $session,
            'POST',
            'https://api.hcaptcha.com/getcaptcha/' . $siteKey,
            '{"pass": true,"generated_pass_UUID": "' . $solution . '","expiration": 120}'
        );

        $this->framesHelper->switchToIFrame($session, $checkboxFrame->getPath());
        $session->executeScript(file_get_contents(__DIR__ . '/../../../Resources/js/hcaptcha.js'));
        $session->executeScript('document.getElementById(\'anchor\').click()');

        $checkVerificationStatusScript = <<<JS
window.hCaptchaSolved === true
JS;
        $passed = $session->wait(5000, $checkVerificationStatusScript);
        Assert::true($passed, 'Checkbox was not checked.');
    }

    public function isVerificationRequired(Session $session): bool
    {
        [$checkboxFrame, $challengeFrame] = $this->getHCaptchaFrames($session);

        return $checkboxFrame && $challengeFrame;
    }

    /**
     * @return array<Frame|null>
     */
    private function getHCaptchaFrames(Session $session): array
    {
        $allFrames = $this->framesHelper->getFrames($session);

        foreach ($allFrames as $frame) {
            if (!str_starts_with($frame->getSrc(), 'https://newassets.hcaptcha.com/captcha/')) {
                continue;
            }

            if (!str_contains($frame->getSrc(), '#frame=checkbox')) {
                continue;
            }

            if ($this->isCaptchaSolved($session, $frame)) {
                continue;
            }

            if (!$this->framesHelper->isFrameInteractable($session, $frame->getPath())) {
                continue;
            }

            $currentCaptchaId = $this->parseQueryValue($frame->getSrc(), 'id');
            Assert::notNull($currentCaptchaId, 'Captcha id was not found.');

            $challengeFrame = $this->getChallengeFrame($session, $currentCaptchaId);

            return [$frame, $challengeFrame];
        }

        return [null, null];
    }

    private function getChallengeFrame(Session $session, string $captchaId): Frame
    {
        $allFrames = $this->framesHelper->getFrames($session);

        foreach ($allFrames as $frame) {
            if (!str_starts_with($frame->getSrc(), 'https://newassets.hcaptcha.com/captcha/')) {
                continue;
            }

            if (!str_contains($frame->getSrc(), '#frame=challenge')) {
                continue;
            }

            $currentCaptchaId = $this->parseQueryValue($frame->getSrc(), 'id');

            if ($currentCaptchaId !== $captchaId) {
                continue;
            }

            return $frame;
        }

        throw new \Exception('Challenge frame was not found.');
    }

    private function parseQueryValue(string $src, string $key): ?string
    {
        $parsedUrl = parse_url($src);
        $query = $parsedUrl['fragment'] ?? '';
        parse_str($query, $params);

        return $params[$key] ?? null;
    }

    private function isCaptchaSolved(Session $session, Frame $frame): bool
    {
        $this->framesHelper->switchToIFrame($session, $frame->getPath());
        $isChallengePassed = $session->evaluateScript('window.hCaptchaSolved === true');
        $this->framesHelper->switchToMainWindow($session);

        return $isChallengePassed;
    }
}
