<?php

namespace Avodel\WebDriver\DependencyInjection;

use Avodel\WebDriver\Components\CaptchaVerifier\Solver\CloudflareTurnstileTokenSolverInterface;
use Avodel\WebDriver\Components\CaptchaVerifier\Solver\HCaptchaSolverInterface;
use Avodel\WebDriver\Components\CaptchaVerifier\Solver\RecaptchaV2SolverInterface;
use Avodel\WebDriver\Driver\MinkPhpWebDriver;
use Avodel\WebDriver\Extension\WebDriverExtensionInterface;
use Avodel\WebDriver\WebDriver;
use Facebook\WebDriver\Chrome\ChromeOptions;
use Symfony\Component\Config\FileLocator;
use Symfony\Component\DependencyInjection\Argument\TaggedIteratorArgument;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Definition;
use Symfony\Component\DependencyInjection\Extension\Extension;
use Symfony\Component\DependencyInjection\Loader\YamlFileLoader;

class AvodelWebDriverExtension extends Extension
{
    public function load(array $configs, ContainerBuilder $container)
    {
        $configuration = new Configuration();
        $config = $this->processConfiguration($configuration, $configs);

        $loader = new YamlFileLoader($container, new FileLocator(__DIR__ . '/../Resources/config'));
        $loader->load('services.yaml');

        $container->registerForAutoconfiguration(WebDriverExtensionInterface::class)->addTag('avodel.web_driver_extension');

        $chromeOptions = new Definition(ChromeOptions::class);
        $chromeOptions->addMethodCall('addArguments', [[
            '--start-maximized',
            '--user-agent=' . $config['user_agent'],
            '--window-size=1920,1080',
            '--disable-blink-features=AutomationControlled',
            '--lang=en-US',
            '--no-sandbox',
        ]]);

        $chromeOptions->addMethodCall('addExtensions', [[
            __DIR__ . '/../Resources/chrome_extensions/i_don_t_care_about_cookies.crx',
        ]]);

        $desiredCapabilities = [
            'acceptSslCerts' => true,
            ChromeOptions::CAPABILITY => $chromeOptions,
        ];

        $webDriver = new Definition(MinkPhpWebDriver::class);
        $webDriver->setArguments([
            '$browserName' => 'chrome',
            '$wdHost' => $config['host'],
            '$desiredCapabilities' => $desiredCapabilities,
            '$proxyUrl' => $config['proxy'],
            '$webDriverExtensions' => new TaggedIteratorArgument('avodel.web_driver_extension')
        ]);

        $WebDriver = new Definition(WebDriver::class);
        $WebDriver->setArguments([
            '$driver' => $webDriver,
        ]);

        $container->setDefinition(WebDriver::class, $WebDriver);

        $solverClasses = $config['components']['captcha_verifier']['solvers'];
        $container->setAlias(HCaptchaSolverInterface::class, $solverClasses['hcaptcha']);
        $container->setAlias(RecaptchaV2SolverInterface::class, $solverClasses['recaptcha_v2']);
        $container->setAlias(CloudflareTurnstileTokenSolverInterface::class, $solverClasses['cloudflare_turnstile']);

        $container->setParameter('avodel.web_driver.ajax.max_waiting_time_ms', $config['components']['ajax']['max_waiting_time_ms']);

        $dumpDir = $config['components']['debug']['dump_dir'];

        if (!is_dir($dumpDir)) {
            throw new \RuntimeException('Dump dir does not exist: ' . $dumpDir);
        }

        $container->setParameter('avodel.web_driver.debug.dump_dir', $dumpDir);
        $container->setParameter('avodel.web_driver.user_agent', $config['user_agent']);
    }
}
